import json
import pathlib
import tempfile
from pathlib import Path
from typing import Any
from typing import assert_never

from loguru import logger

from imbue_core.sculptor.state.mcp_constants import IMBUE_CLI_INTERNAL_MCP_SERVER_NAME
from imbue_core.sculptor.state.mcp_constants import IMBUE_CLI_USER_MCP_SERVER_NAME
from sculptor.interfaces.environments.base import Environment
from sculptor.services.config_service.data_types import AWSBedrockApiKey
from sculptor.services.config_service.data_types import AnthropicApiKey
from sculptor.services.config_service.data_types import AnthropicCredentials
from sculptor.services.config_service.data_types import ClaudeOauthCredentials

# Number of characters from the end of API key to store for approval tracking
API_KEY_SUFFIX_LENGTH = 20


def populate_claude_settings(environment: Environment, anthropic_credentials: AnthropicCredentials) -> None:
    """Claude Code requires certain settings to run correctly.

    We default to using the user's settings (with some specific changes).
    However, if the user does NOT have claude code installed, we can provide
    them with our own settings.
    """
    logger.info("Populating claude settings")
    claude_config_path = Path.home() / ".claude.json"

    if claude_config_path.exists():
        logger.info("Found existing claude config path at {}", claude_config_path)
        claude_config = json.load(open(claude_config_path, "r"))

        # Make required modifications
        claude_config["projects"][str(environment.to_host_path(environment.get_workspace_path()))] = (
            _claude_project_config(environment.to_host_path(environment.get_workspace_path()))
        )
    else:
        logger.info("Generating new claude config")
        claude_config = _claude_config_template(environment)
    # A previous version of this code set the primaryApiKey,
    # but we have since moved to injecting environment variables.
    # Remove it in case the container has an old config
    # and there's a conflict between the two approaches.
    claude_config.pop("primaryApiKey", None)
    credentials_file_path = environment.get_container_user_home_directory() / ".claude" / ".credentials.json"
    match anthropic_credentials:
        case AnthropicApiKey(anthropic_api_key=anthropic_api_key):
            # this is required for claude to work with the anthropic api key without prompting the user (primarily required for compaction and terminal)
            claude_config["customApiKeyResponses"] = {
                "approved": [anthropic_api_key.unwrap()[-API_KEY_SUFFIX_LENGTH:]],
                "rejected": [],
            }
            logger.trace("Writing anthropic api key to {}", credentials_file_path)
            # TODO: This works but it's safer to remove the file completely.
            environment.write_file(str(credentials_file_path), "")
            logger.trace("Wrote anthropic api key to {}", credentials_file_path)
        case ClaudeOauthCredentials():
            claude_config["customApiKeyResponses"] = {
                "approved": [],
                "rejected": [],
            }
            logger.trace("Writing claude oauth credentials to {}", credentials_file_path)
            environment.write_file(
                str(credentials_file_path), anthropic_credentials.convert_to_claude_code_credentials_json()
            )
            logger.trace("Wrote claude oauth credentials to {}", credentials_file_path)
        case AWSBedrockApiKey(bedrock_api_key=bedrock_api_key):
            claude_config["customApiKeyResponses"] = {
                "approved": [bedrock_api_key.unwrap()[-API_KEY_SUFFIX_LENGTH:]],
                "rejected": [],
            }
            logger.trace("Writing bedrock api key to {}", credentials_file_path)
            # TODO: This works but it's safer to remove the file completely.
            environment.write_file(str(credentials_file_path), "")
            logger.trace("Wrote bedrock api key to {}", credentials_file_path)
        case _ as unreachable:
            # TODO: pyre doesn't understand the matching here
            assert_never(unreachable)  # pyre-fixme[6]
    claude_config["hasCompletedOnboarding"] = True

    with tempfile.NamedTemporaryFile() as tmp_file:
        tmp_file.write(json.dumps(claude_config).encode())
        tmp_file.flush()
        claude_config_path = environment.get_container_user_home_directory() / ".claude.json"
        logger.trace("Copying claude config to environment at {}", claude_config_path)
        environment.copy_from_local(Path(tmp_file.name), str(claude_config_path))

    logger.info("Populated claude settings")


def _claude_mcp_servers_config(workspace_path: pathlib.Path) -> dict[str, Any]:
    return {
        name: {
            "command": "imbue-cli.sh",
            "args": [
                "--log-to-file=/tmp/imbue-cli.log",
                "mcp",
                *("--project-path", str(workspace_path)),
                *config_args,
                *("--transport", "stdio"),
            ],
            "env": {},
        }
        for name, config_args in (
            (IMBUE_CLI_INTERNAL_MCP_SERVER_NAME, ["--use-internal-config"]),
            (IMBUE_CLI_USER_MCP_SERVER_NAME, ["--config", str(workspace_path / "tools.toml")]),
        )
    }


def _claude_project_config(workspace_path: pathlib.Path) -> dict[str, Any]:
    # TODO: do we need all of these settings? last session id seems to be randomly copy pasted from someone's .claude.json
    return {
        "allowedTools": [],
        "history": [],
        "dontCrawlDirectory": False,
        "mcpContextUris": [],
        "mcpServers": _claude_mcp_servers_config(workspace_path),
        "enabledMcpjsonServers": [],
        "disabledMcpjsonServers": [],
        "hasTrustDialogAccepted": True,
        "ignorePatterns": [],
        "projectOnboardingSeenCount": 1,
        "hasClaudeMdExternalIncludesApproved": False,
        "hasClaudeMdExternalIncludesWarningShown": False,
        "lastCost": 0,
        "lastAPIDuration": 0,
        "lastDuration": 3172,
        "lastLinesAdded": 0,
        "lastLinesRemoved": 0,
        "lastTotalInputTokens": 0,
        "lastTotalOutputTokens": 0,
        "lastTotalCacheCreationInputTokens": 0,
        "lastTotalCacheReadInputTokens": 0,
        "lastSessionId": "ef949ec0-4a45-4665-81a7-f9e1ec21a41c",
        "bypassPermissionsModeAccepted": True,
    }


def _claude_config_template(environment: Environment) -> dict[str, Any]:
    return {
        "numStartups": 3,
        "theme": "light",
        "customApiKeyResponses": {
            "approved": [],
            "rejected": [],
        },
        "firstStartTime": "2025-06-10T21:50:05.520Z",
        "projects": {
            str(environment.to_host_path(environment.get_workspace_path())): _claude_project_config(
                environment.to_host_path(environment.get_workspace_path())
            )
        },
        "isQualifiedForDataSharing": False,
        "hasCompletedOnboarding": True,
        "lastOnboardingVersion": "1.0.17",
        "recommendedSubscription": "",
        "subscriptionNoticeCount": 0,
        "hasAvailableSubscription": False,
    }
