"""Uses the Codex scripting mode to run a Codex agent."""

import json
import shlex
from typing import assert_never

from loguru import logger
from pydantic import PrivateAttr

from imbue_core.agents.data_types.ids import TaskID
from imbue_core.sculptor.state.messages import ChatInputUserMessage
from imbue_core.sculptor.state.messages import Message
from sculptor.agents.default.agent_wrapper import DefaultAgentWrapper
from sculptor.agents.default.claude_code_sdk.agent_wrapper import on_git_user_message
from sculptor.agents.default.claude_code_sdk.artifact_creation import get_file_artifact_messages
from sculptor.agents.default.codex.process_manager import CodexProcessManager
from sculptor.agents.default.codex.utils import populate_codex_settings
from sculptor.agents.default.constants import DEFAULT_WAIT_TIMEOUT
from sculptor.agents.default.constants import GITLAB_PROJECT_URL_STATE_FILE
from sculptor.agents.default.constants import GITLAB_TOKEN_STATE_FILE
from sculptor.agents.default.constants import REMOVED_MESSAGE_IDS_STATE_FILE
from sculptor.interfaces.agents.agent import CodexAgentConfig
from sculptor.interfaces.agents.agent import CommandInputUserMessage
from sculptor.interfaces.agents.agent import CompactTaskUserMessage
from sculptor.interfaces.agents.agent import ForkAgentSystemMessage
from sculptor.interfaces.agents.agent import GitCommitAndPushUserMessage
from sculptor.interfaces.agents.agent import GitPullUserMessage
from sculptor.interfaces.agents.agent import InterruptProcessUserMessage
from sculptor.interfaces.agents.agent import LocalSyncDisabledMessage
from sculptor.interfaces.agents.agent import LocalSyncSetupAndEnabledMessage
from sculptor.interfaces.agents.agent import LocalSyncSetupProgressMessage
from sculptor.interfaces.agents.agent import LocalSyncSetupStartedMessage
from sculptor.interfaces.agents.agent import LocalSyncUpdateCompletedMessage
from sculptor.interfaces.agents.agent import LocalSyncUpdatePausedMessage
from sculptor.interfaces.agents.agent import LocalSyncUpdatePendingMessage
from sculptor.interfaces.agents.agent import ManualSyncMergeIntoAgentAttemptedMessage
from sculptor.interfaces.agents.agent import MessageFeedbackUserMessage
from sculptor.interfaces.agents.agent import RemoveQueuedMessageAgentMessage
from sculptor.interfaces.agents.agent import RemoveQueuedMessageUserMessage
from sculptor.interfaces.agents.agent import RequestSkippedAgentMessage
from sculptor.interfaces.agents.agent import ResumeAgentResponseRunnerMessage
from sculptor.interfaces.agents.agent import SetProjectConfigurationDataUserMessage
from sculptor.interfaces.agents.agent import SetUserConfigurationDataUserMessage
from sculptor.interfaces.agents.agent import StopAgentUserMessage
from sculptor.interfaces.agents.artifacts import ArtifactType
from sculptor.interfaces.agents.constants import AGENT_EXIT_CODE_CLEAN_SHUTDOWN_ON_INTERRUPT
from sculptor.interfaces.agents.constants import AGENT_EXIT_CODE_SHUTDOWN_DUE_TO_EXCEPTION
from sculptor.services.config_service.data_types import AnthropicCredentials


class CodexAgent(DefaultAgentWrapper):
    config: CodexAgentConfig
    task_id: TaskID
    _codex_process_manager: CodexProcessManager | None = PrivateAttr(default=None)

    def _push_message(self, message: Message) -> None:
        match message:
            case RemoveQueuedMessageUserMessage():
                with self._handle_user_message(message):
                    self._removed_message_ids.add(message.target_message_id.suffix)
                    self.environment.write_file(
                        str(self.environment.get_state_path() / REMOVED_MESSAGE_IDS_STATE_FILE),
                        json.dumps(list(self._removed_message_ids)),
                    )
                    logger.info("Removed message id: {}", message.target_message_id)
                    self._output_messages.put(
                        RemoveQueuedMessageAgentMessage(removed_message_id=message.target_message_id)
                    )
            case CommandInputUserMessage() | ChatInputUserMessage() | ResumeAgentResponseRunnerMessage():
                if message.message_id.suffix in self._removed_message_ids:
                    logger.info("Skipping message {} as it has been removed", message.message_id)
                    self._output_messages.put(
                        RequestSkippedAgentMessage(request_id=message.message_id)  # pyre-fixme[28]
                    )
                    return
                assert self._codex_process_manager is not None, "Codex process manager must be set"
                self._codex_process_manager.process_input_message(message)
            case LocalSyncUpdateCompletedMessage() | ManualSyncMergeIntoAgentAttemptedMessage():
                logger.info("Received local sync update message, updating artifacts")
                messages_to_send = get_file_artifact_messages(
                    artifact_name=ArtifactType.DIFF,
                    environment=self.environment,
                    source_branch=self.source_branch,
                    task_id=self.task_id,
                )
                for artifact_message in messages_to_send:
                    self._output_messages.put(artifact_message)
            case (
                LocalSyncSetupStartedMessage()
                | LocalSyncSetupProgressMessage()
                | LocalSyncSetupAndEnabledMessage()
                | LocalSyncUpdatePendingMessage()
                | LocalSyncUpdatePausedMessage()
                | LocalSyncDisabledMessage()
            ):
                pass
            # TODO: eventually just make this GitCommitUserMessage
            case GitCommitAndPushUserMessage():
                with self._handle_user_message(message):
                    commit_message = shlex.quote(message.commit_message)
                    task_branch = shlex.quote(self.task_branch)
                    commit_and_push_command_string = f"if [ \"$(git branch --show-current)\" != {task_branch} ]; then echo 'Error: Current branch is not {task_branch}'; exit 1; fi && git add . && git commit -m {commit_message} --trailer 'Co-authored-by: Sculptor <sculptor@imbue.com>'"
                    # when settings.IS_NEW_MANUAL_SYNC_ENABLED is true, we do not want to push
                    if message.is_pushing:
                        commit_and_push_command_string += " && git push sculptor"
                    on_git_user_message(
                        environment=self.environment,
                        command=["bash", "-c", commit_and_push_command_string],
                        source_branch=self.source_branch,
                        output_message_queue=self._output_messages,
                        task_id=self.task_id,
                    )
            case GitPullUserMessage():
                with self._handle_user_message(message):
                    on_git_user_message(
                        environment=self.environment,
                        command=["git", "pull"],
                        source_branch=self.source_branch,
                        output_message_queue=self._output_messages,
                        task_id=self.task_id,
                    )
            # FIXME: make an error message for local sync
            case CompactTaskUserMessage():
                # CODEX TODO
                assert self._codex_process_manager is not None, "Codex process manager must be set"
                self._codex_process_manager.process_input_message(message)
            case InterruptProcessUserMessage():
                assert self._codex_process_manager is not None, "Codex process manager must be set"
                self._codex_process_manager.interrupt_current_message(message)
            case StopAgentUserMessage():
                logger.info("Stopping agent")
                with self._handle_user_message(message):
                    self.terminate(DEFAULT_WAIT_TIMEOUT)
                    self._exit_code = AGENT_EXIT_CODE_CLEAN_SHUTDOWN_ON_INTERRUPT
                # # it doesn't make sense to make markers for starting and stopping this message
                # # since the semantics are that, once stopping, we no longer get the RequestSuccessAgentMessage messages
                # # so we very specifically do NOT call `self._handle_user_message(message)` here
                # self._is_stopping = True
                # self.terminate(DEFAULT_WAIT_TIMEOUT)
                # self._exit_code = AGENT_EXIT_CODE_CLEAN_SHUTDOWN_ON_INTERRUPT
                logger.info("Finished stopping agent")
            case SetUserConfigurationDataUserMessage():
                # CODEX TODO
                logger.info("User configuration message received")
                self._load_credentials(anthropic_credentials=None)
            case SetProjectConfigurationDataUserMessage():
                logger.info("Project configuration message received")
                self.environment.write_file(
                    str(self.environment.get_state_path() / GITLAB_TOKEN_STATE_FILE), message.gitlab_token
                )
                self.environment.write_file(
                    str(self.environment.get_state_path() / GITLAB_PROJECT_URL_STATE_FILE), message.gitlab_url
                )
            case MessageFeedbackUserMessage():
                logger.info("Message feedback received for message {}", message.feedback_message_id)
                pass
            case ForkAgentSystemMessage():
                pass
            case _ as unreachable:
                # pyre-fixme[6]: The type of message should be changed to an appropriate union type instead of a base type
                assert_never(unreachable)

    def terminate(self, force_kill_seconds: float = 5.0) -> None:
        # Stop the terminal manager first
        if self._terminal_manager:
            self._terminal_manager.stop()

        # CODEX TODO
        assert self._codex_process_manager is not None, "Codex process manager must be set"
        self._codex_process_manager.stop(force_kill_seconds, is_waiting=False)

    def poll(self) -> int | None:
        assert self._codex_process_manager is not None, "Codex process manager must be set"
        if self._codex_process_manager.get_exception_if_exists() is not None:
            self._exit_code = AGENT_EXIT_CODE_SHUTDOWN_DUE_TO_EXCEPTION
        return super().poll()

    def wait(self, timeout: float) -> int:
        assert self._codex_process_manager is not None, "Codex process manager must be set"
        self._codex_process_manager.stop(timeout, is_waiting=True)

        assert self._exit_code is not None, (
            "The wait method will only ever terminate if the agent is stopped or if there is an exception"
        )
        return self._exit_code

    def _start(self) -> None:
        # Initialize the Claude process manager
        self._codex_process_manager = CodexProcessManager(
            environment=self.environment,
            in_testing=self.in_testing,
            secrets=self._secrets,
            task_id=self.task_id,
            output_message_queue=self._output_messages,
            handle_user_message_callback=self._handle_user_message,
            system_prompt=self.system_prompt,
            source_branch=self.source_branch,
            task_branch=self.task_branch,
        )

    # TODO CODEX: it's a little weird that this is called from both inside and outside, on both codex and claude
    def _load_credentials(self, anthropic_credentials: AnthropicCredentials | None) -> None:
        populate_codex_settings(environment=self.environment)
