import shutil
import threading
from pathlib import Path
from tempfile import TemporaryDirectory
from typing import Final

from imbue_core.processes.local_process import run_blocking
from sculptor.utils.build import get_sculptor_folder
from sculptor.utils.file_utils import copy_dir


def copy_ssh_config(dest: Path):
    # Location of this file (ssh_utils.py)
    here = Path(__file__).resolve().parent

    # Source directory: adjacent ssh_config
    src = here / "ssh_config"

    # Make sure destination exists
    dest.mkdir(parents=True, exist_ok=True)

    # Copy everything (overwrite if exists)
    for item in src.iterdir():
        target = dest / item.name
        if item.is_dir():
            # Replace dir if already exists
            if target.exists():
                shutil.rmtree(target)
            copy_dir(item, target)
        else:
            shutil.copy2(item, target)


_SSH_CONFIGURATION_LOCK: Final[threading.Lock] = threading.Lock()


def ensure_local_sculptor_ssh_configured() -> Path:
    dot_sculptor = get_sculptor_folder()
    copy_ssh_config(dot_sculptor / "ssh")

    keypair_directory = dot_sculptor / "task_container_keypair"
    keypair_directory.mkdir(parents=True, exist_ok=True)

    dest_private = keypair_directory / "id_rsa"
    dest_public = keypair_directory / "id_rsa.pub"

    with _SSH_CONFIGURATION_LOCK:
        private_exists = dest_private.exists()
        public_exists = dest_public.exists()

        if private_exists and public_exists:
            return keypair_directory

        with TemporaryDirectory(dir=keypair_directory) as temp_dir_str:
            temp_dir = Path(temp_dir_str)
            temp_private = temp_dir / "id_rsa"
            temp_public = temp_dir / "id_rsa.pub"

            run_blocking(["ssh-keygen", "-t", "rsa", "-f", str(temp_private), "-N", ""])

            temp_private.replace(dest_private)
            temp_public.replace(dest_public)

    return keypair_directory
