import os
import time
from pathlib import Path
from threading import Event
from typing import Mapping
from typing import Sequence

import httpx
from dotenv import dotenv_values
from dotenv import set_key
from loguru import logger
from pydantic import ValidationError

from imbue_core.async_monkey_patches import log_exception
from imbue_core.gitlab_management import GITLAB_TOKEN_NAME
from imbue_core.sculptor.user_config import UserConfig
from imbue_core.secrets_utils import Secret
from sculptor.primitives.threads import ObservableThread
from sculptor.services.config_service.api import ConfigService
from sculptor.services.config_service.data_types import AnthropicApiKey
from sculptor.services.config_service.data_types import AnthropicCredentials
from sculptor.services.config_service.data_types import CLAUDE_CODE_CLIENT_ID
from sculptor.services.config_service.data_types import ClaudeOauthCredentials
from sculptor.services.config_service.data_types import Credentials
from sculptor.services.config_service.data_types import REFRESH_TOKEN_EXPIRY_BUFFER_SECONDS
from sculptor.services.config_service.data_types import TokenResponse
from sculptor.services.config_service.data_types import UserConfiguration
from sculptor.services.config_service.user_config import get_config_path
from sculptor.services.config_service.user_config import get_user_config_instance
from sculptor.services.config_service.user_config import save_config
from sculptor.services.config_service.user_config import set_user_config_instance
from sculptor.services.config_service.utils import populate_anthropic_credentials_file
from sculptor.utils.build import get_sculptor_folder


class LocalConfigService(ConfigService):
    secret_file_path: Path = get_sculptor_folder() / ".env"
    credentials_file_path: Path = get_sculptor_folder() / "credentials.json"
    # FIXME: these should all be private vars?
    anthropic_credentials: AnthropicCredentials | None = None
    token_refresh_stop_event: Event = Event()
    token_refresh_thread: ObservableThread | None = None

    def start(self) -> None:
        self.secret_file_path.parent.mkdir(parents=True, exist_ok=True)
        try:
            credentials = Credentials.model_validate_json(self.credentials_file_path.read_text())
            self.set_anthropic_credentials(credentials.anthropic)
        except (FileNotFoundError, ValidationError):
            user_config = get_user_config_instance()
            if user_config and user_config.anthropic_api_key:
                self.set_anthropic_credentials(
                    AnthropicApiKey(
                        anthropic_api_key=Secret(user_config.anthropic_api_key), generated_from_oauth=False
                    )
                )

    def stop(self) -> None:
        if self.token_refresh_thread:
            self._stop_token_refresh_thread()

    def get_anthropic_credentials(self) -> AnthropicCredentials | None:
        return self.anthropic_credentials

    def set_anthropic_credentials(self, anthropic_credentials: AnthropicCredentials) -> None:
        old_credentials_is_claude_oauth = isinstance(self.anthropic_credentials, ClaudeOauthCredentials)
        new_credentials_is_claude_oauth = isinstance(anthropic_credentials, ClaudeOauthCredentials)
        if old_credentials_is_claude_oauth and not new_credentials_is_claude_oauth:
            self._stop_token_refresh_thread()
        self.anthropic_credentials = anthropic_credentials
        if isinstance(anthropic_credentials, ClaudeOauthCredentials):
            self._on_new_user_config(UserConfiguration(anthropic_credentials=self.anthropic_credentials))
        populate_anthropic_credentials_file(self.credentials_file_path, anthropic_credentials)
        if not old_credentials_is_claude_oauth and new_credentials_is_claude_oauth:
            self._start_token_refresh_thread()

    def _start_token_refresh_thread(self) -> None:
        self.token_refresh_thread = self.concurrency_group.start_new_thread(target=self._token_refresh_thread_target)

    def _stop_token_refresh_thread(self) -> None:
        self.token_refresh_stop_event.set()
        self.token_refresh_thread.join()  # pyre-fixme[16]: token_refresh_thread can be None
        self.token_refresh_thread = None
        self.token_refresh_stop_event = Event()

    def _token_refresh_thread_target(self) -> None:
        first_iteration = True
        while not self.concurrency_group.is_shutting_down():
            if first_iteration:
                first_iteration = False
            else:
                # Wait for a short time between all iterations,
                # but not before the first iteration -
                # the OAuth token might already have expired when Sculptor starts.
                #
                # The timeout may seem unnecessarily short short,
                # as the token is usually valid for at least a couple of hours.
                # However, the user's computer could go to sleep and we can overshoot the expiry.
                # Minimize that possiblity by checking more frequently.
                should_stop = self.token_refresh_stop_event.wait(timeout=30)
                if should_stop:
                    break
            logger.debug("Claude OAuth token refresh thread has woken up")
            anthropic_credentials = self.anthropic_credentials
            assert isinstance(anthropic_credentials, ClaudeOauthCredentials)
            if time.time() < anthropic_credentials.expires_at_unix_ms / 1000 - REFRESH_TOKEN_EXPIRY_BUFFER_SECONDS:
                continue
            logger.info("Refreshing Claude OAuth tokens")
            refresh_token = anthropic_credentials.refresh_token.unwrap()
            with httpx.Client() as client:
                try:
                    raw_response = client.post(
                        "https://console.anthropic.com/v1/oauth/token",
                        data={
                            "grant_type": "refresh_token",
                            "refresh_token": refresh_token,
                            "client_id": CLAUDE_CODE_CLIENT_ID,
                        },
                        headers={"Accept": "application/json"},
                    )
                    token_response = TokenResponse.model_validate_json(raw_response.content)
                except Exception as e:
                    log_exception(e, "Error refreshing Claude OAuth credentials")
                    # If we have failed, the response wouldn't contain any secret credentials,
                    # so it's safe to log.
                    logger.info("Raw response: {}", raw_response.content)
                    logger.info("Ignoring the error; we'll try again later")
                    continue
            self.anthropic_credentials = ClaudeOauthCredentials(
                access_token=Secret(token_response.access_token),
                refresh_token=Secret(token_response.refresh_token),
                expires_at_unix_ms=int((time.time() + token_response.expires_in) * 1000),
                scopes=token_response.scope.split(" "),
                subscription_type=anthropic_credentials.subscription_type,
            )
            populate_anthropic_credentials_file(self.credentials_file_path, self.anthropic_credentials)
            self._on_new_user_config(UserConfiguration(anthropic_credentials=self.anthropic_credentials))

    def _on_new_user_config(self, user_config: UserConfiguration) -> None:
        pass
        # if self.configuration_broadcast_service:
        #     self.configuration_broadcast_service.broadcast_configuration_to_all_tasks(user_config)

    def remove_anthropic_credentials(self) -> None:
        self.anthropic_credentials = None
        try:
            self.credentials_file_path.unlink()
        except FileNotFoundError:
            pass

    def get_user_secrets(self, secret_names: Sequence[str] | None = None) -> dict[str, Secret]:
        file_secrets = {}
        if self.secret_file_path.exists():
            file_secrets = dotenv_values(self.secret_file_path)

        secrets = file_secrets
        if os.getenv(GITLAB_TOKEN_NAME) is not None:
            # If the user has this token in their environment, propagate it into the agent.
            secrets[GITLAB_TOKEN_NAME] = os.environ[GITLAB_TOKEN_NAME]

        if secret_names is not None:
            secrets = {name: secrets[name] for name in secret_names if name in secrets}

        secrets = {key: Secret(value) for key, value in secrets.items()}

        return secrets

    def set_user_secrets(self, secrets: Mapping[str, str | Secret]) -> None:
        logger.debug("Saving {} secrets to {}", len(secrets), self.secret_file_path)

        self.secret_file_path.parent.mkdir(parents=True, exist_ok=True)

        for key, value in secrets.items():
            set_key(
                dotenv_path=str(self.secret_file_path),
                key_to_set=key,
                value_to_set=value.unwrap() if isinstance(value, Secret) else value,
                quote_mode="auto",
            )

    def get_user_config(self) -> UserConfig:
        return get_user_config_instance()

    def set_user_config(self, config: UserConfig) -> None:
        config_path = get_config_path()
        save_config(config, config_path)
        set_user_config_instance(config)
