from enum import StrEnum
from pathlib import Path
from typing import Final

import json5
from loguru import logger

from imbue_core.agents.data_types.ids import ProjectID
from imbue_core.agents.data_types.ids import TaskID
from imbue_core.async_monkey_patches import log_exception
from imbue_core.background_setup import BackgroundSetup
from imbue_core.concurrency_group import ConcurrencyGroup
from imbue_core.sculptor import telemetry
from imbue_core.sculptor.telemetry import PosthogEventModel
from imbue_core.sculptor.telemetry import PosthogEventPayload
from imbue_core.sculptor.telemetry import emit_posthog_event
from imbue_core.sculptor.telemetry import with_consent
from imbue_core.sculptor.telemetry_constants import ConsentLevel
from imbue_core.sculptor.telemetry_constants import ProductComponent
from imbue_core.sculptor.telemetry_constants import SculptorPosthogEvent
from imbue_core.thread_utils import ObservableThread
from sculptor.interfaces.environments.base import LocalDevcontainerImageConfig
from sculptor.interfaces.environments.base import LocalDockerImage
from sculptor.services.environment_service.api import DEFAULT_TASK_SPECIFIC_CONTEXT
from sculptor.services.environment_service.api import TaskSpecificContext
from sculptor.services.environment_service.providers.docker.image_fetch import ImagePurpose
from sculptor.services.environment_service.providers.docker.image_fetch import fetch_image_from_cdn
from sculptor.services.environment_service.providers.docker.image_utils import build_docker_image
from sculptor.services.environment_service.providers.docker.volume_mounted_nix_control_plane import (
    CONTROL_PLANE_FETCH_BACKGROUND_SETUP,
)
from sculptor.utils.timeout import log_runtime_decorator

_IMBUE_ADDONS_DOCKERFILE_PATH: Final[Path] = Path(__file__).parent / "imbue_addons" / "Dockerfile.imbue_addons"


class DevcontainerBuildPath(StrEnum):
    """Control flow paths for devcontainer image building."""

    DOCKERFILE_NAME = "dockerfile_name"
    IMAGE_NAME = "image_name"
    FALLBACK_TO_DEFAULT = "fallback_to_default"


class DevcontainerBuildEventData(PosthogEventPayload):
    """PostHog event data for devcontainer build operations."""

    control_flow_path: str = with_consent(ConsentLevel.PRODUCT_ANALYTICS)
    devcontainer_json_path: str = with_consent(ConsentLevel.PRODUCT_ANALYTICS)
    tag: str = with_consent(ConsentLevel.PRODUCT_ANALYTICS)
    fallback_reason: str | None = with_consent(ConsentLevel.PRODUCT_ANALYTICS)


class DevcontainerError(ValueError):
    """Error raised when there's an issue with the DevcontainerError."""

    pass


def get_default_devcontainer_json_path() -> Path:
    result = Path(__file__).parent / "default_devcontainer" / "devcontainer.json"
    assert result.exists(), f"Default devcontainer.json not found at {result}"
    return result


def get_default_devcontainer_image_reference() -> str:
    """Parse and return the image reference from the default devcontainer.json."""
    default_devcontainer_path = get_default_devcontainer_json_path()
    json_contents = json5.loads(default_devcontainer_path.read_text("utf-8"))
    image_reference = json_contents.get("image")
    assert image_reference, f"No 'image' field found in default devcontainer.json at {default_devcontainer_path}"
    return image_reference


@log_runtime_decorator()
def docker_pull_default_devcontainer(concurrency_group: ConcurrencyGroup) -> None:
    """Download and ensure default devcontainer image is available.

    This function downloads the image tarball to cache, then ensures it's loaded into Docker.
    """
    image_reference = get_default_devcontainer_image_reference()
    # Try to fetch the devcontainer image from CDN first
    logger.info("Starting download and load of default devcontainer image: {}", image_reference)
    fetch_image_from_cdn(image_reference, ImagePurpose.DEFAULT_DEVCONTAINER, concurrency_group)


PULL_DEFAULT_DEVCONTAINER_BACKGROUND_SETUP: Final[BackgroundSetup] = BackgroundSetup(
    "DockerPullDefaultDevcontainerBackgroundSetup",
    docker_pull_default_devcontainer,
)


def start_control_plane_background_setup(
    thread_suffix: str, concurrency_group: ConcurrencyGroup
) -> list[ObservableThread]:
    """Starting control plane background setup tasks.  Does not block, just starts background threads."""
    logger.info("Starting background setup tasks for devcontainers.")
    return [
        PULL_DEFAULT_DEVCONTAINER_BACKGROUND_SETUP.start_run_in_background(
            thread_name=f"DockerPullDefaultDevcontainerBackgroundSetup_{thread_suffix}",
            concurrency_group=concurrency_group,
        ),
        CONTROL_PLANE_FETCH_BACKGROUND_SETUP.start_run_in_background(
            thread_name=f"ControlPlaneFetchBackgroundSetup_{thread_suffix}", concurrency_group=concurrency_group
        ),
    ]


def get_devcontainer_json_path_from_repo_or_default(repo_path: Path) -> Path:
    """Find the user's devcontainer.json file, or use our default one so they don't have to specify it."""
    paths = [
        ".devcontainer/devcontainer.json",
        "devcontainer.json",
    ]
    for p in paths:
        if (repo_path / p).exists():
            logger.info("Found devcontainer.json at {}", p)
            return repo_path / p
    result = get_default_devcontainer_json_path()
    logger.info("No devcontainer.json found, using the Sculptor default at {}", result)
    return result


class DefaultDevcontainerFallbackPayload(PosthogEventPayload):
    """PostHog event data for default devcontainer fallback."""

    reason: str = with_consent(ConsentLevel.ERROR_REPORTING)


@log_runtime_decorator()
def build_local_devcontainer_image(
    config: LocalDevcontainerImageConfig,
    cached_repo_tarball_parent_directory: Path,
    project_id: ProjectID,
    tag: str,
    concurrency_group: ConcurrencyGroup,
    task_id: TaskID | None = None,
    secrets: dict[str, str] | None = None,
    task_specific_context: TaskSpecificContext = DEFAULT_TASK_SPECIFIC_CONTEXT,
) -> LocalDockerImage:
    """Build a Docker image from a devcontainer.json configuration."""
    logger.info("Building local devcontainer image from {} with tag {}", config.devcontainer_json_path, tag)

    # Start control plane volume setup in background thread
    control_plane_thread = CONTROL_PLANE_FETCH_BACKGROUND_SETUP.start_run_in_background(
        thread_name="ControlPlaneFetchJoinedThread", concurrency_group=concurrency_group
    )

    devcontainer_path = Path(config.devcontainer_json_path)
    if not devcontainer_path.exists():
        raise FileNotFoundError(f"devcontainer.json not found at {devcontainer_path}")

    # Initialize container_user to None so it's always defined
    container_user: str | None = None

    try:
        json_contents = json5.loads(devcontainer_path.read_text("utf-8"))
        # TODO: Consider somehow invoking the reference implementation via:
        # devcontainer build --workspace-folder devcontainer_path.parent.
        # For now, we are just supporting a very limited amount of the devcontainer.json format.

        # Parse the containerUser field if present
        # See: https://containers.dev/implementors/spec/#users
        container_user = json_contents.get("containerUser")

        # We support two different ways to build a devcontainer image:
        # 1. From a Dockerfile: devcontainer.json's build.dockerfile field
        # 2. From an image: devcontainer.json's image field
        # Exactly one of these must be specified, and we check this.
        dockerfile_name = json_contents.get("build", {}).get("dockerfile")
        image_name = json_contents.get("image")
        if not dockerfile_name and not image_name:
            raise DevcontainerError(
                f"devcontainer.json must contain a 'build.dockerfile' field or an 'image' field, {json_contents=}"
            )
        elif dockerfile_name and image_name:
            raise DevcontainerError(
                f"devcontainer.json cannot contain both a 'build.dockerfile' field and an 'image' field, {json_contents=}"
            )
        # Initialize PostHog event data - control_flow_path and fallback_reason will be set in the branches
        control_flow_path: DevcontainerBuildPath
        fallback_reason: str | None = None

        if dockerfile_name:
            build_context = json_contents.get("build", {}).get("context", ".")
            build_context_path = devcontainer_path.parent / build_context
            # Build from a Dockerfile
            dockerfile_path = devcontainer_path.parent / dockerfile_name
            if not dockerfile_path.exists():
                raise DevcontainerError(f"Dockerfile not found at {dockerfile_path}")

            user_image_tag = f"{tag}_user_image_to_wrap"

            logger.info(
                "Building user image from Dockerfile at {}, with build context at {}",
                dockerfile_path,
                build_context_path,
            )
            user_image: LocalDockerImage = build_docker_image(
                dockerfile_path,
                project_id=project_id,
                concurrency_group=concurrency_group,
                tag=user_image_tag,
                build_path=build_context_path,
                secrets=secrets,
            )
            if task_id is not None:
                telemetry.emit_posthog_event(
                    PosthogEventModel(
                        name=SculptorPosthogEvent.ENVIRONMENT_SETUP_LOCAL_DOCKERFILE_BUILT,
                        component=ProductComponent.ENVIRONMENT_SETUP,
                        task_id=str(task_id),
                    )
                )
            logger.info("Built user image tag with tag={}, id={}", user_image_tag, user_image.image_id)
            control_flow_path = DevcontainerBuildPath.DOCKERFILE_NAME
        else:
            # Use the pre-existing image.
            # The great thing about this path is that it skips an entire docker build step.
            assert image_name is not None
            user_image_tag = image_name
            control_flow_path = DevcontainerBuildPath.IMAGE_NAME
    except Exception as e:
        # TODO: Somehow get a message into Sculptor's message queue with the logs from the failure.
        log_exception(e, "Failed to build user Dockerfile, falling back to default devcontainer image")
        fallback_reason = f"Dockerfile build failed: {type(e).__name__}"

        task_specific_context.emit_warning(
            "Failed to build devcontainer image from Dockerfile, falling back to default devcontainer image. Check the logs tab for additional details."
        )
        if task_id is not None:
            telemetry.emit_posthog_event(
                PosthogEventModel(
                    name=SculptorPosthogEvent.ENVIRONMENT_SETUP_FELL_BACK_TO_DEFAULT_DEVCONTAINER,
                    component=ProductComponent.ENVIRONMENT_SETUP,
                    task_id=str(task_id),
                    payload=DefaultDevcontainerFallbackPayload(reason=fallback_reason),
                )
            )

        # Fall back to using the default devcontainer image
        user_image_tag = get_default_devcontainer_image_reference()
        control_flow_path = DevcontainerBuildPath.FALLBACK_TO_DEFAULT

    logger.info("Building Imbue's wrapper image around user_image_tag={}", user_image_tag)
    try:
        wrapped_image: LocalDockerImage = build_docker_image(
            _IMBUE_ADDONS_DOCKERFILE_PATH,
            project_id=project_id,
            concurrency_group=concurrency_group,
            cached_repo_tarball_parent_directory=cached_repo_tarball_parent_directory,
            tag=tag,
            secrets=secrets,
            base_image_tag=user_image_tag,
            container_user=container_user,
        )
        if task_id is not None:
            telemetry.emit_posthog_event(
                PosthogEventModel(
                    name=SculptorPosthogEvent.ENVIRONMENT_SETUP_WRAPPER_DOCKERFILE_BUILT,
                    component=ProductComponent.ENVIRONMENT_SETUP,
                    task_id=str(task_id),
                )
            )
        logger.info("Built Imbue's wrapper image with tag={}", tag)
    except Exception as e:
        log_exception(
            e,
            "Failed to build Imbue's wrapper around user_image_tag={user_image_tag}, falling back to default devcontainer image.",
            user_image_tag=user_image_tag,
        )

        task_specific_context.emit_warning(
            "Failed to build Imbue's wrapper around your devcontainer image, falling back to default devcontainer image. Check the logs tab for additional details."
        )

        if task_id is not None:
            telemetry.emit_posthog_event(
                PosthogEventModel(
                    name=SculptorPosthogEvent.ENVIRONMENT_SETUP_FELL_BACK_TO_DEFAULT_DEVCONTAINER,
                    component=ProductComponent.ENVIRONMENT_SETUP,
                    task_id=str(task_id),
                    payload=DefaultDevcontainerFallbackPayload(reason=f"Imbue wrapper build failed: {repr(e)[:200]}"),
                )
            )
        # The reason this is almost repeated is to handle the case where devcontainer.json specifies an image,
        # but the image is not valid.  In that case, there's no build step, for the user image, but the
        # build_docker_image above for _IMBUE_ADDONS_DOCKERFILE_PATH will fail, and we fall back to using
        # the default devcontainer image.
        wrapped_image: LocalDockerImage = build_docker_image(
            _IMBUE_ADDONS_DOCKERFILE_PATH,
            project_id=project_id,
            concurrency_group=concurrency_group,
            cached_repo_tarball_parent_directory=cached_repo_tarball_parent_directory,
            tag=tag,
            secrets=secrets,
            base_image_tag=get_default_devcontainer_image_reference(),
            container_user=container_user,
        )
        if task_id is not None:
            telemetry.emit_posthog_event(
                PosthogEventModel(
                    name=SculptorPosthogEvent.ENVIRONMENT_SETUP_WRAPPER_DOCKERFILE_BUILT,
                    component=ProductComponent.ENVIRONMENT_SETUP,
                    task_id=str(task_id),
                )
            )
        logger.info("As a fallback, built Imbue's wrapper image with tag={}", tag)
        control_flow_path = DevcontainerBuildPath.FALLBACK_TO_DEFAULT

    # Emit PostHog telemetry event
    try:
        event_data = DevcontainerBuildEventData(
            control_flow_path=control_flow_path,
            devcontainer_json_path=str(devcontainer_path),
            tag=tag,
            fallback_reason=fallback_reason,
        )
        posthog_event = PosthogEventModel[
            DevcontainerBuildEventData
        ](
            name=SculptorPosthogEvent.TASK_START_MESSAGE,  # Using existing event - could add DEVCONTAINER_BUILD if needed
            component=ProductComponent.TASK,
            payload=event_data,
        )
        emit_posthog_event(posthog_event)
    except Exception as e:
        logger.info("Failed to emit devcontainer build telemetry: {}", e)

    if task_id is not None:
        telemetry.emit_posthog_event(
            PosthogEventModel(
                name=SculptorPosthogEvent.ENVIRONMENT_SETUP_WAITING_FOR_CONTROL_PLANE_SETUP,
                component=ProductComponent.ENVIRONMENT_SETUP,
                task_id=str(task_id),
            )
        )
    # Wait for control plane thread to complete and raise any errors
    control_plane_thread.join()  # This will raise any exception from the background thread

    # The container_user has been baked into the image at /imbue_addons/container_user.txt
    # during the docker build process, so we don't need to return it separately.
    return wrapped_image
