from abc import ABC
from abc import abstractmethod
from pathlib import Path
from typing import Callable
from typing import Mapping
from typing import Sequence

from imbue_core.agents.data_types.ids import ProjectID
from imbue_core.sculptor.user_config import UserConfig
from imbue_core.secrets_utils import Secret
from sculptor.primitives.service import Service
from sculptor.services.config_service.data_types import AnthropicCredentials
from sculptor.services.config_service.data_types import GlobalAnthropicConfiguration
from sculptor.services.config_service.data_types import ProjectConfiguration


class ConfigService(Service, ABC):
    @abstractmethod
    def get_anthropic_credentials(self) -> AnthropicCredentials | None: ...

    @abstractmethod
    def set_anthropic_credentials(self, anthropic_credentials: AnthropicCredentials):
        """
        Set Anthropic credentials.

        If the credentials are ClaudeOauthCredentials,
        the service is also responsible for refreshing them.
        """

    @abstractmethod
    def remove_anthropic_credentials(self):
        """Remove the stored Anthropic credentials, and delete the credentials file if it exists."""

    @abstractmethod
    def get_user_secrets(self, secret_names: Sequence[str] | None) -> dict[str, Secret]:
        """
        Retrieve secrets by their names.

        :param secret_names: List of secret names to retrieve.  If None, all secrets should be returned.
        :return: Dictionary mapping secret names to their values.
        """

    @abstractmethod
    def set_user_secrets(self, secrets: Mapping[str, str | Secret]) -> None:
        """
        Saves all secrets.
        """

    @abstractmethod
    def get_user_config(self) -> UserConfig:
        """
        Retrieve the current user configuration.
        """

    @abstractmethod
    def set_user_config(self, config: UserConfig) -> None:
        """
        Set the current user configuration.
        """

    @abstractmethod
    def get_global_anthropic_configuration(self) -> GlobalAnthropicConfiguration:
        """
        Retrieve the current global (user-level) Anthropic configuration.

        """

    @abstractmethod
    def register_global_anthropic_configuration_watcher(
        self, callback: Callable[[GlobalAnthropicConfiguration], None]
    ) -> None:
        """
        NOTE: Currently only watches changes to GlobalAnthropicConfiguration.claude_config

        """

    @abstractmethod
    def register_project_configuration_watcher(
        self, project_id: "ProjectID", workspace_path: Path, callback: Callable[[ProjectConfiguration], None]
    ) -> None:
        """
        NOTE: Currently only watches changes to ProjectConfiguration.claude_workspace_settings

        """
