import threading
from contextlib import contextmanager
from typing import Generator


class SharedExclusiveLock:
    """A shared/exclusive lock that allows multiple sharers or a single excluder.

    Sharers (sometimes called readers) share the lock unless an excluder (writer) is active or waiting.
    Excluders gain exclusive access and block new sharers until they release the lock.

    AKA: ReadWriteLock.
    """

    def __init__(self) -> None:
        self._condition = threading.Condition()
        self._active_sharers = 0
        self._active_excluder = False
        self._waiting_excluders = 0

    @contextmanager
    def shared_lock(self) -> Generator[None, None, None]:
        """Acquire the lock which can be shared by multiple threads, unless an excluder is active."""
        with self._condition:
            while self._waiting_excluders > 0 or self._active_excluder:
                self._condition.wait()
            self._active_sharers += 1
        try:
            yield
        finally:
            with self._condition:
                self._active_sharers -= 1
                if self._active_sharers == 0:
                    self._condition.notify_all()

    @contextmanager
    def exclusive_lock(self) -> Generator[None, None, None]:
        """Acquire an exclusive lock which can be held by at most one thread and excludes shared locks."""
        with self._condition:
            self._waiting_excluders += 1
            while self._active_excluder or self._active_sharers > 0:
                self._condition.wait()
            self._active_excluder = True
            self._waiting_excluders -= 1
        try:
            yield
        finally:
            with self._condition:
                self._active_excluder = False
                self._condition.notify_all()
