import json
import pathlib
import shlex
import tempfile
from pathlib import Path
from typing import Any
from typing import Final
from typing import Mapping
from typing import Sequence
from typing import assert_never

from loguru import logger

from imbue_core.sculptor.state.mcp_constants import IMBUE_CLI_INTERNAL_MCP_SERVER_NAME
from imbue_core.sculptor.state.mcp_constants import IMBUE_CLI_USER_MCP_SERVER_NAME
from sculptor.agents.data_types import SlashCommand
from sculptor.agents.default.claude_code_sdk.constants import CLAUDE_DIRECTORY
from sculptor.agents.default.claude_code_sdk.constants import CLAUDE_JSON_FILENAME
from sculptor.agents.default.claude_code_sdk.constants import COMMANDS_DIRECTORY
from sculptor.agents.default.claude_code_sdk.constants import CREDENTIALS_JSON_FILENAME
from sculptor.database.models import Project
from sculptor.interfaces.environments.base import Environment
from sculptor.services.config_service.api import ConfigService
from sculptor.services.config_service.data_types import AWSBedrockApiKey
from sculptor.services.config_service.data_types import AnthropicApiKey
from sculptor.services.config_service.data_types import AnthropicCredentials
from sculptor.services.config_service.data_types import ClaudeOauthCredentials
from sculptor.services.environment_service.tool_readiness import READY_FILE

# Number of characters from the end of API key to store for approval tracking
API_KEY_SUFFIX_LENGTH = 20

PROJECT_SCOPE_COMMAND_UI_MARKER = "(project)"
USER_SCOPE_COMMAND_UI_MARKER = "(user)"

SUPPORTED_BUILTIN_SLASH_COMMANDS = ("compact",)


def populate_claude_settings(
    project: Project,
    environment: Environment,
    anthropic_credentials: AnthropicCredentials,
    config_service: ConfigService,
) -> None:
    """
    Populate .claude.json and .credentials.json in the environment.

    (.claude/{settings.json,subagents,commands} are populated via a different mechanism - see the ConfigService.)

    Claude Code requires certain settings to run correctly.

    We default to using the user's settings (with some specific changes).
    However, if the user does NOT have claude code installed, we can provide
    them with our own settings.

    """
    logger.info("Populating claude settings")
    claude_config_path_local = config_service.get_claude_root_path_on_users_local_machine() / CLAUDE_JSON_FILENAME
    claude_credentials_path_local = (
        config_service.get_claude_root_path_on_users_local_machine() / CLAUDE_DIRECTORY / CREDENTIALS_JSON_FILENAME
    )
    workspace_key = str(environment.to_host_path(environment.get_workspace_path()))

    if claude_config_path_local.exists():
        logger.info("Found existing claude config path at {}", claude_config_path_local)
        claude_config = json.load(open(claude_config_path_local, "r"))
        local_project_path = project.get_local_user_path().expanduser().resolve()
        local_project_configuration = (
            claude_config.get("projects", {}).get(str(local_project_path))
            if config_service.get_user_config().is_claude_configuration_synchronized
            else None
        )
        claude_config["projects"] = {}  # Clear out projects that only exist on the user's machine.
        claude_config["projects"][workspace_key] = _claude_project_config(
            environment.to_host_path(environment.get_workspace_path()), local_project_configuration
        )
        config_service.maybe_refresh_claude_mcp_oauth_tokens(local_project_path)
    else:
        logger.info("Generating new claude config")
        claude_config = _claude_config_template(environment)
    # A previous version of this code set the primaryApiKey,
    # but we have since moved to injecting environment variables.
    # Remove it in case the container has an old config
    # and there's a conflict between the two approaches.
    claude_config.pop("primaryApiKey", None)
    credentials_file_path = (
        environment.get_container_user_home_directory() / CLAUDE_DIRECTORY / CREDENTIALS_JSON_FILENAME
    )
    credentials_data = {}
    custom_api_key_responses, claude_ai_credentials = _custom_api_key_responses_and_claude_ai_credentials(
        anthropic_credentials
    )
    if claude_ai_credentials is not None:
        credentials_data.update(claude_ai_credentials)
    mcp_servers_oauth = _claude_mcp_servers_oauth(claude_credentials_path_local)
    if mcp_servers_oauth is not None:
        credentials_data.update(mcp_servers_oauth)

    logger.trace("Writing claude oauth credentials to {}", credentials_file_path)
    if len(credentials_data) == 0:
        # TODO: This works but it's safer to remove the file completely.
        environment.write_atomically(str(credentials_file_path), ".tmp", "")
    else:
        environment.write_atomically(str(credentials_file_path), ".tmp", json.dumps(credentials_data))

    claude_config["customApiKeyResponses"] = custom_api_key_responses
    claude_config["hasCompletedOnboarding"] = True
    with tempfile.NamedTemporaryFile() as tmp_file:
        claude_config_path = environment.get_container_user_home_directory() / CLAUDE_JSON_FILENAME
        logger.trace("Writing claude config to environment at {}", claude_config_path)
        environment.write_atomically(str(claude_config_path), ".tmp", json.dumps(claude_config))

    logger.info("Populated claude settings")


def _custom_api_key_responses_and_claude_ai_credentials(
    anthropic_credentials: AnthropicCredentials,
) -> tuple[dict[str, Any], dict[str, Any] | None]:
    match anthropic_credentials:
        case AnthropicApiKey(anthropic_api_key=anthropic_api_key):
            # this is required for claude to work with the anthropic api key without prompting the user (primarily required for compaction and terminal)
            return {
                "approved": [anthropic_api_key.unwrap()[-API_KEY_SUFFIX_LENGTH:]],
                "rejected": [],
            }, None
        case ClaudeOauthCredentials():
            return {
                "approved": [],
                "rejected": [],
            }, anthropic_credentials.convert_to_claude_code_credentials_json_section()
        case AWSBedrockApiKey(bedrock_api_key=bedrock_api_key):
            return {
                "approved": [bedrock_api_key.unwrap()[-API_KEY_SUFFIX_LENGTH:]],
                "rejected": [],
            }, None
        case _ as unreachable:
            # TODO: pyre doesn't understand the matching here
            assert_never(unreachable)  # pyre-fixme[6]


def _claude_mcp_servers_config(
    workspace_path: pathlib.Path, user_local_config: dict[str, Any] | None
) -> dict[str, Any]:
    mcp_servers = {
        name: {
            "command": "imbue-cli.sh",
            "args": [
                "--log-to-file=/tmp/imbue-cli.log",
                "mcp",
                *("--project-path", str(workspace_path)),
                *config_args,
                *("--transport", "stdio"),
            ],
            "env": {},
        }
        for name, config_args in (
            (IMBUE_CLI_INTERNAL_MCP_SERVER_NAME, ["--use-internal-config"]),
            (IMBUE_CLI_USER_MCP_SERVER_NAME, ["--config", str(workspace_path / "tools.toml")]),
        )
    }
    if user_local_config is not None:
        for server_name, server_config in user_local_config.get("mcpServers", {}).items():
            if server_name not in mcp_servers:
                mcp_servers[server_name] = server_config
    return mcp_servers


def _claude_mcp_servers_oauth(credentials_json_local_path: Path) -> dict[str, Any] | None:
    if not credentials_json_local_path.exists():
        return None
    with open(credentials_json_local_path, "r") as f:
        credentials_data = json.load(f)
    if "mcpOAuth" not in credentials_data:
        return None
    mcp_oauth_data = credentials_data["mcpOAuth"]
    for server_name, server_details in mcp_oauth_data.items():
        # Remove refresh tokens.
        # (We don't want them to be propagated to the sandboxes, otherwise claude instances in the sandboxes would invalidate them by using them.)
        server_details.pop("refreshToken", None)
    return {
        "mcpOAuth": mcp_oauth_data,
    }


def _claude_project_config(workspace_path: pathlib.Path, user_local_config: dict[str, Any] | None) -> dict[str, Any]:
    # TODO: do we need all of these settings? last session id seems to be randomly copy pasted from someone's .claude.json
    return {
        "allowedTools": [],
        "history": [],
        "dontCrawlDirectory": False,
        "mcpContextUris": user_local_config.get("mcpContextUris", []) if user_local_config is not None else [],
        # NOTE: Additional MCP servers are added later on through SetProjectConfigurationDataUserMessages.
        "mcpServers": _claude_mcp_servers_config(workspace_path, user_local_config),
        "enabledMcpjsonServers": [],
        "disabledMcpjsonServers": [],
        "hasTrustDialogAccepted": True,
        "ignorePatterns": [],
        "projectOnboardingSeenCount": 1,
        "hasClaudeMdExternalIncludesApproved": False,
        "hasClaudeMdExternalIncludesWarningShown": False,
        "lastCost": 0,
        "lastAPIDuration": 0,
        "lastDuration": 3172,
        "lastLinesAdded": 0,
        "lastLinesRemoved": 0,
        "lastTotalInputTokens": 0,
        "lastTotalOutputTokens": 0,
        "lastTotalCacheCreationInputTokens": 0,
        "lastTotalCacheReadInputTokens": 0,
        "lastSessionId": "ef949ec0-4a45-4665-81a7-f9e1ec21a41c",
        "bypassPermissionsModeAccepted": True,
    }


def _claude_config_template(environment: Environment) -> dict[str, Any]:
    return {
        "numStartups": 3,
        "theme": "light",
        "customApiKeyResponses": {
            "approved": [],
            "rejected": [],
        },
        "firstStartTime": "2025-06-10T21:50:05.520Z",
        "projects": {
            str(environment.to_host_path(environment.get_workspace_path())): _claude_project_config(
                environment.to_host_path(environment.get_workspace_path()), None
            )
        },
        "isQualifiedForDataSharing": False,
        "hasCompletedOnboarding": True,
        "lastOnboardingVersion": "1.0.17",
        "recommendedSubscription": "",
        "subscriptionNoticeCount": 0,
        "hasAvailableSubscription": False,
    }


def get_all_supported_slash_commands(environment: Environment) -> tuple[SlashCommand, ...]:
    """
    Return all custom commands found in the project and user home directories as well as the supported built-in commands.

    The ordering, value as well as display name are all consistent with Claude Code's behavior.

    """
    command_groups: list[list[SlashCommand]] = []
    for commands_directory_path, suffix in (
        (environment.get_workspace_path() / CLAUDE_DIRECTORY / COMMANDS_DIRECTORY, PROJECT_SCOPE_COMMAND_UI_MARKER),
        (
            environment.get_container_user_home_directory() / CLAUDE_DIRECTORY / COMMANDS_DIRECTORY,
            USER_SCOPE_COMMAND_UI_MARKER,
        ),
    ):
        command_group: list[SlashCommand] = []
        find_command = [
            "find",
            str(commands_directory_path),
            "-name",
            "*.md",
            "-type",
            "f",
        ]
        process = environment.run_process_to_completion(find_command, secrets={}, is_checked=False)
        assert process.returncode in (0, 1)
        lines = process.stdout
        run_paths = []
        for line in lines.splitlines():
            line = line.strip()
            if line:
                command_path = Path(line).relative_to(commands_directory_path)
                command_string = command_path.stem
                for parent in command_path.parents:
                    if parent.stem:
                        command_string = f"{parent.stem}:{command_string}"

                command_group.append(
                    SlashCommand(
                        value=f"/{command_string}",
                        display_name=f"{command_string} {suffix}",
                    )
                )
        command_groups.append(command_group)

    # Claude Code makes the built-in commands part of the last group.
    command_groups[-1].extend(
        SlashCommand(value=f"/{command_string}", display_name=command_string)
        for command_string in SUPPORTED_BUILTIN_SLASH_COMMANDS
    )
    for command_group in command_groups:
        command_group.sort(key=lambda command: command.value)
    return tuple(command for group in command_groups for command in group)


def _read_existing_settings(environment: Environment, settings_path: pathlib.Path) -> dict[str, Any]:
    """Read existing settings.json file if it exists.

    Args:
        environment: The environment to read from
        settings_path: Path to the settings.json file

    Returns:
        Existing settings, or empty dict if file doesn't exist

    Raises:
        Exception: If an unexpected error occurs (not FileNotFoundError)
        json.JSONDecodeError: If the existing settings file has invalid JSON
    """
    try:
        content = environment.read_file(str(settings_path))
        logger.debug("Read existing settings file from {}", settings_path)
    except FileNotFoundError:
        logger.debug("No existing settings file at {}", settings_path)
        return {}

    settings = json.loads(content)
    logger.debug("Parsed settings with {} top-level keys", len(settings))
    return settings


_HOOK_SCRIPT_PATH: Final[str] = "/imbue/bin/check_tool_readiness.sh"


def _matches_tool_readiness_hook(hook_config: object) -> bool:
    """Check if a given hook configuration matches our tool readiness hook."""

    # Imperative logic to check if the structure looks something like this:
    # {
    #   "matcher": "*",
    #   "hooks": [
    #     {
    #       "type": "command",
    #       "command": "/imbue/bin/check_tool_readiness.sh <ready_file_path>",
    #       ...,
    #     },
    #   ],
    # }
    #
    # Since this can contain arbitrary user-defined data, we have to be defensive.

    if not isinstance(hook_config, Mapping):
        return False
    matcher = hook_config.get("matcher")
    hooks = hook_config.get("hooks")
    if matcher != "*":
        return False
    if not isinstance(hooks, Sequence) or len(hooks) == 0:
        return False
    first_hook = hooks[0]
    if not isinstance(first_hook, Mapping):
        return False
    command = first_hook.get("command")
    if not isinstance(command, str):
        return False
    return _HOOK_SCRIPT_PATH in command


def _merge_tool_readiness_hook(existing_settings: Mapping[str, Any], timeout_seconds: int) -> dict[str, Any]:
    """Merge tool readiness hook configuration with existing settings.

    Preserves all existing settings and hooks, appending the tool readiness PreToolUse hook.
    Multiple hooks with the same matcher can coexist and will all run in parallel.

    Args:
        existing_settings: Existing settings
        timeout_seconds: Timeout for the tool readiness hook

    Returns:
        New settings dictionary with hook appended

    Raises:
        TypeError: If existing settings structure has unexpected types
    """
    our_hook_config = {
        "matcher": "*",
        "hooks": [
            {
                "type": "command",
                "command": f"{_HOOK_SCRIPT_PATH} {shlex.quote(READY_FILE.as_posix())}",
                "env": {
                    "SCULPTOR_TOOL_READINESS_TIMEOUT": str(timeout_seconds),
                },
            }
        ],
    }

    existing_hooks = existing_settings.get("hooks", {})
    if not isinstance(existing_hooks, dict):
        raise TypeError(f"Expected hooks to be a dict, got {type(existing_hooks).__name__}")

    existing_pre_tool_use = existing_hooks.get("PreToolUse", [])
    if not isinstance(existing_pre_tool_use, (list, tuple)):
        raise TypeError(f"Expected PreToolUse to be a list, got {type(existing_pre_tool_use).__name__}")

    logger.debug("Appending tool readiness hook to PreToolUse hooks")

    return {
        **existing_settings,
        "hooks": {
            **existing_hooks,
            "PreToolUse": [
                *(
                    hook_config
                    for hook_config in existing_pre_tool_use
                    if not _matches_tool_readiness_hook(hook_config)
                ),
                our_hook_config,
            ],
        },
    }


def configure_tool_readiness_hook(environment: Environment, timeout_seconds: int = 120) -> None:
    """Configure the PreToolUse hook to block tool execution until environment is ready.

    Merges the hook configuration with any existing settings.json, preserving all
    other settings and hooks.

    Args:
        environment: The environment to configure
        timeout_seconds: Maximum seconds to wait before timing out
    """
    logger.info("Configuring tool readiness hook (timeout: {}s)", timeout_seconds)

    settings_dir = environment.get_container_user_home_directory() / ".claude"
    settings_path = settings_dir / "settings.json"

    existing_settings = _read_existing_settings(environment, settings_path)
    if not isinstance(existing_settings, dict):
        raise TypeError(f"Expected existing settings to be a dict, got {type(existing_settings).__name__}")
    merged_settings = _merge_tool_readiness_hook(existing_settings, timeout_seconds)

    with tempfile.NamedTemporaryFile(mode="w", suffix=".json") as tmp_file:
        json.dump(merged_settings, tmp_file, indent=2)
        tmp_file.flush()

        environment.run_process_to_completion(["mkdir", "-p", str(settings_dir)], is_checked=True, secrets={})
        environment.copy_from_local(Path(tmp_file.name), str(settings_path))
        logger.info("Tool readiness hook configured successfully at {}", settings_path)
