import json
from datetime import datetime
from datetime import timedelta
from datetime import timezone

from loguru import logger

from imbue_core.concurrency_group import ConcurrencyGroup
from imbue_core.subprocess_utils import ProcessError


class ControlPlaneVolumeGarbageCollector:
    def __init__(self, latest_volume_name: str, concurrency_group: ConcurrencyGroup) -> None:
        self.latest_volume_name = latest_volume_name
        self.concurrency_group = concurrency_group

    def _volume_is_in_use(self, volume_name: str) -> bool:
        """Check if a Docker volume is currently in use by any container."""
        ps_result = self.concurrency_group.run_process_to_completion(
            command=[
                "docker",
                "ps",
                "-a",
                "--filter",
                f"volume={volume_name}",
                "--format",
                "{{.ID}}",
            ]
        )
        containers_using_volume = [c for c in ps_result.stdout.strip().split("\n") if c]
        return len(containers_using_volume) > 0

    def _get_volume_creation_date(self, volume_name: str) -> datetime | None:
        """Get the creation date of a Docker volume.

        Returns None if the creation date cannot be determined.
        """
        try:
            inspect_result = self.concurrency_group.run_process_to_completion(
                command=["docker", "volume", "inspect", volume_name, "--format", "{{json .}}"]
            )
            volume_info = json.loads(inspect_result.stdout.strip())

            created_at_str = volume_info.get("CreatedAt", "")
            if not created_at_str:
                logger.debug(f"Could not determine creation date for volume {volume_name}")
                return None

            # Parse the timestamp (format: 2024-01-15T10:30:45Z or 2024-01-15T10:30:45-07:00)
            try:
                created_at = datetime.fromisoformat(created_at_str.replace("Z", "+00:00"))
                return created_at
            except ValueError:
                logger.debug(f"Could not parse creation date for volume {volume_name}: {created_at_str}")
                return None
        except ProcessError as e:
            logger.debug(f"Failed to inspect volume {volume_name}: {e}")
            return None

    def _should_prune_volume(self, volume_name: str) -> bool:
        # Skip the current volume
        if volume_name == self.latest_volume_name:
            logger.debug(f"Skipping current volume: {volume_name}")
            return False

        # Check if volume is in use
        if self._volume_is_in_use(volume_name):
            return False

        # Check creation time
        created_at = self._get_volume_creation_date(volume_name)
        if created_at is None:
            return False

        # Only prune volumes created MORE than 7 days ago (created_at < cutoff_date)
        cutoff_date = datetime.now(timezone.utc) - timedelta(days=7)
        if created_at > cutoff_date:
            return False

        return True

    def _get_volumes_to_prune(self) -> list[str]:
        # List all volumes that start with imbue_control_plane_
        result = self.concurrency_group.run_process_to_completion(
            command=[
                "docker",
                "volume",
                "ls",
                "--format",
                "{{.Name}}",
            ]
        )

        all_volumes = result.stdout.strip().split("\n")
        control_plane_volumes = [v for v in all_volumes if v.startswith("imbue_control_plane_")]
        logger.debug(f"Found {len(control_plane_volumes)} control plane volumes")

        volumes_to_prune = [v for v in control_plane_volumes if self._should_prune_volume(v)]
        logger.info(f"Found {len(volumes_to_prune)} pruneable control plane volumes")

        return volumes_to_prune

    def _prune_volumes(self, volumes_to_prune: list[str]) -> None:
        """Delete the specified Docker volumes."""
        if not volumes_to_prune:
            logger.debug("No old control plane volumes to prune")
            return

        logger.debug(f"Pruning {len(volumes_to_prune)} control plane volumes.")
        successfully_pruned = 0
        for volume_name in volumes_to_prune:
            try:
                self.concurrency_group.run_process_to_completion(command=["docker", "volume", "rm", volume_name])
                logger.debug(f"Successfully pruned volume: {volume_name}")
                successfully_pruned += 1
            except ProcessError as e:
                logger.debug(f"Failed to prune volume {volume_name}: {e}")
        logger.info(f"Pruned {successfully_pruned} control plane volumes.")

    def prune_old_control_plane_volumes(self) -> None:
        """Deletes old control plane Docker volumes that match all of the following criteria:
        - name starts with imbue_control_plane_
        - is not the current/latest version
        - is not in use
        - was created >7 days ago (time limit to make it easy to switch between release channels or versions; we may tighten this in the future)
        """
        try:
            volumes_to_prune = self._get_volumes_to_prune()
            self._prune_volumes(volumes_to_prune)

        except Exception as e:
            # Don't fail the entire operation if pruning fails, but log a warning
            logger.debug(f"Error during control plane volume pruning: {e}")
