from typing import Callable

from loguru import logger

from imbue_core.errors import ExpectedError
from imbue_core.sculptor.telemetry import send_exception_to_posthog
from imbue_core.sculptor.telemetry_constants import SculptorPosthogEvent
from sculptor.services.task_service.data_types import ServiceCollectionForTask


def run_cleanup_images_task_v1(
    services: ServiceCollectionForTask, on_started: Callable[[], None] | None = None
) -> None:
    """Run the cleanup images task."""
    if on_started is not None:
        on_started()
    logger.debug("Starting Docker image cleanup process")
    try:
        deleted_images = services.environment_service.remove_stale_images()
    except ExpectedError as e:
        if "Docker daemon is not running" in str(e):
            send_exception_to_posthog(
                SculptorPosthogEvent.TASK_FAILED_WITH_EXPECTED_ERROR,
                e,
                include_traceback=True,
            )
            logger.info("Docker daemon is not running, skipping image cleanup")
            return
        raise

    # Log the results
    logger.info("Docker image cleanup completed. Deleted {} images", len(deleted_images))
    if deleted_images:
        logger.trace("Deleted image IDs: {}", deleted_images)
