from abc import ABC
from abc import abstractmethod
from pathlib import Path
from typing import Callable
from typing import Mapping
from typing import Sequence

from imbue_core.agents.data_types.ids import ProjectID
from imbue_core.errors import ExpectedError
from imbue_core.sculptor.user_config import UserConfig
from imbue_core.secrets_utils import Secret
from sculptor.primitives.service import Service
from sculptor.services.config_service.data_types import AnthropicCredentials
from sculptor.services.config_service.data_types import Credentials
from sculptor.services.config_service.data_types import GlobalConfiguration
from sculptor.services.config_service.data_types import OpenAIApiKey
from sculptor.services.config_service.data_types import ProjectConfiguration


class MissingCredentialsError(ExpectedError):
    pass


class ConfigService(Service, ABC):
    @abstractmethod
    def get_credentials(self) -> Credentials: ...

    @abstractmethod
    def set_anthropic_credentials(self, anthropic_credentials: AnthropicCredentials) -> None:
        """
        Set Anthropic credentials.

        If the credentials are ClaudeOauthCredentials,
        the service is also responsible for refreshing them.
        """

    @abstractmethod
    def remove_anthropic_credentials(self) -> None:
        """Remove the stored Anthropic credentials."""

    @abstractmethod
    def set_openai_credentials(self, openai_credentials: OpenAIApiKey) -> None:
        """
        Set OpenAI credentials.
        """

    @abstractmethod
    def remove_openai_credentials(self) -> None:
        """Remove the stored OpenAI credentials."""

    @abstractmethod
    def get_user_secrets(self, secret_names: Sequence[str] | None) -> dict[str, Secret]:
        """
        Retrieve secrets by their names.

        :param secret_names: List of secret names to retrieve.  If None, all secrets should be returned.
        :return: Dictionary mapping secret names to their values.
        """

    @abstractmethod
    def set_user_secrets(self, secrets: Mapping[str, str | Secret]) -> None:
        """
        Saves all secrets.
        """

    @abstractmethod
    def get_user_config(self) -> UserConfig:
        """
        Retrieve the current user configuration.
        """

    @abstractmethod
    def set_user_config(self, config: UserConfig) -> None:
        """
        Set the current user configuration.
        """

    @abstractmethod
    def get_global_configuration(self) -> GlobalConfiguration:
        """
        Retrieve the current global (user-level) configuration.

        (NOTE: .claude.json settings are managed through claude_code_sdk/config_utils.py.)

        """

    @abstractmethod
    def get_claude_root_path_on_users_local_machine(self) -> Path:
        """
        Where .claude and .claude.json are expected to be found.

        This is typically the user's home directory but we allow overriding it for testing purposes.

        """

    @abstractmethod
    def register_global_configuration_watcher(
        self, callback: Callable[[GlobalConfiguration, str | None], None]
    ) -> None:
        """
        NOTE: Currently only watches changes to GlobalConfiguration.claude_config

        """

    @abstractmethod
    def register_project_configuration_watcher(
        self,
        project_id: "ProjectID",
        workspace_path: Path,
        callback: Callable[[ProjectConfiguration, str | None], None],
    ) -> None:
        """
        NOTE: Currently only watches changes to ProjectConfiguration.claude_workspace_settings

        """

    @abstractmethod
    def maybe_refresh_claude_mcp_oauth_tokens(self, workspace_path: Path) -> None:
        """
        Refresh the Claude MCP OAuth tokens if they are near expiration and if user has claude locally installed.

        NOTE: we currently do not have a mechanism for refreshing tokens inside a Sculptor sandbox in case of long-running tasks.
        We should add that in the future after the form of config service evolves further.

        """
