import json
import os
import time
from datetime import datetime
from datetime import timedelta
from pathlib import Path
from threading import Event
from threading import Lock
from typing import Any
from typing import Callable
from typing import Mapping
from typing import Sequence

import httpx
from dotenv import dotenv_values
from dotenv import set_key
from loguru import logger
from pydantic import Field
from pydantic import PrivateAttr
from pydantic import ValidationError
from watchdog.events import FileSystemEvent
from watchdog.events import FileSystemEventHandler
from watchdog.observers import Observer

from imbue_core.agents.data_types.ids import ProjectID
from imbue_core.async_monkey_patches import log_exception
from imbue_core.gitlab_management import GITLAB_TOKEN_NAME
from imbue_core.sculptor.user_config import UserConfig
from imbue_core.secrets_utils import Secret
from imbue_core.subprocess_utils import ProcessSetupError
from sculptor.agents.default.claude_code_sdk.config_utils import get_local_credentials_data
from sculptor.agents.default.claude_code_sdk.config_utils import merge_tool_readiness_hook
from sculptor.agents.default.claude_code_sdk.constants import CLAUDE_DIRECTORY
from sculptor.agents.default.claude_code_sdk.constants import CLAUDE_GLOBAL_SETTINGS_FILENAME
from sculptor.agents.default.claude_code_sdk.constants import CLAUDE_LOCAL_SETTINGS_FILENAME
from sculptor.agents.default.claude_code_sdk.constants import COMMANDS_DIRECTORY
from sculptor.agents.default.claude_code_sdk.constants import SUBAGENTS_DIRECTORY
from sculptor.primitives.threads import ObservableThread
from sculptor.services.config_service.api import ConfigService
from sculptor.services.config_service.data_types import AnthropicApiKey
from sculptor.services.config_service.data_types import AnthropicCredentials
from sculptor.services.config_service.data_types import CLAUDE_CODE_CLIENT_ID
from sculptor.services.config_service.data_types import ClaudeGlobalConfiguration
from sculptor.services.config_service.data_types import ClaudeOauthCredentials
from sculptor.services.config_service.data_types import Credentials
from sculptor.services.config_service.data_types import GlobalConfiguration
from sculptor.services.config_service.data_types import OpenAIApiKey
from sculptor.services.config_service.data_types import ProjectConfiguration
from sculptor.services.config_service.data_types import REFRESH_TOKEN_EXPIRY_BUFFER_SECONDS
from sculptor.services.config_service.data_types import TokenResponse
from sculptor.services.config_service.user_config import get_config_path
from sculptor.services.config_service.user_config import get_user_config_instance
from sculptor.services.config_service.user_config import save_config
from sculptor.services.config_service.user_config import set_user_config_instance
from sculptor.services.config_service.utils import populate_credentials_file
from sculptor.utils.build import get_sculptor_folder

# How long before the OAuth token expiry we should typically attempt to refresh.
CLAUDE_MCP_OAUTH_BUFFER_LONG_SECONDS = 20 * 60
# It can happen that some of the tokens are so shortlived (e.g. 15 mins) that we would always fall into the buffer period.
# We try not to refresh more often than this threshold.
CLAUDE_MCP_OAUTH_FREQUENCY_THRESHOLD_SECONDS = 5 * 60
# We would still refresh the token if it's expired or almost expired, though.
CLAUDE_MCP_OAUTH_BUFFER_SHORT_SECONDS = 2 * 60


class _ClaudeConfigFileHandler(FileSystemEventHandler):
    def __init__(self, config_service: "LocalConfigService"):
        self.config_service = config_service

    def _on_modified(self, event_path: Path) -> None:
        raise NotImplementedError()

    def on_modified(self, event: FileSystemEvent) -> None:
        if event.is_directory:
            return
        assert isinstance(event.src_path, str), "Expected src_path to be a string"
        return self._on_modified(Path(event.src_path))


class _ClaudeGlobalSettingsChangeHandler(_ClaudeConfigFileHandler):
    def _on_modified(self, event_path: Path) -> None:
        if event_path.name == CLAUDE_GLOBAL_SETTINGS_FILENAME:
            is_changed = self.config_service.load_claude_global_settings()
            update_description = None if not is_changed else "Claude settings.json updated."
            self.config_service.notify_global_anthropic_configuration_watchers(update_description)
        elif event_path.suffix == ".md" and event_path.parent.name == SUBAGENTS_DIRECTORY:
            is_changed = self.config_service.load_agent_configurations()
            update_description = None if not is_changed else "Claude subagents updated."
            self.config_service.notify_global_anthropic_configuration_watchers(update_description)
        elif event_path.suffix == ".md" and COMMANDS_DIRECTORY in (p.name for p in event_path.parents):
            is_changed = self.config_service.load_custom_command_configurations()
            update_description = None if not is_changed else "Claude custom commands updated."
            self.config_service.notify_global_anthropic_configuration_watchers(update_description)


class _ClaudeLocalSettingsChangeHandler(_ClaudeConfigFileHandler):
    def _on_modified(self, event_path: Path) -> None:
        if event_path.name != CLAUDE_LOCAL_SETTINGS_FILENAME:
            return
        workspace_path = event_path.parent.parent
        is_changed = self.config_service.load_claude_local_settings(workspace_path)
        update_description = None if not is_changed else "Local Claude settings updated."
        self.config_service.notify_project_configuration_watchers(workspace_path, update_description)


class LocalConfigService(ConfigService):
    secret_file_path: Path = Field(default_factory=lambda: get_sculptor_folder() / ".env")
    credentials_file_path: Path = Field(default_factory=lambda: get_sculptor_folder() / "credentials.json")
    # Where do we expect .claude and .claude.json to be found on the user's local machine.
    claude_root_path_on_users_local_machine: Path
    # FIXME: these should all be private vars?
    credentials: Credentials = Field(default_factory=Credentials)
    token_refresh_stop_event: Event = Field(default_factory=Event)
    token_refresh_thread: ObservableThread | None = None

    _settings_lock: Lock = PrivateAttr(default_factory=Lock)
    _claude_global_settings: dict[str, Any] | None = PrivateAttr(default=None)
    _claude_local_settings: dict[str, dict[str, Any] | None] = PrivateAttr(default_factory=dict)
    _claude_global_subagents: dict[str, str] | None = PrivateAttr(default=None)
    # A mapping from file path relative to ~/.claude/commands/ to command definition.
    _claude_global_custom_commands: dict[str, str] | None = PrivateAttr(default=None)
    _anthropic_global_configuration_watchers: list[Callable[[GlobalConfiguration, str | None], None]] = PrivateAttr(
        default_factory=list
    )
    _observer: Observer | None = PrivateAttr(default=None)  # pyre-fixme[11]
    _project_configuration_watchers: dict[
        str, tuple["ProjectID", Callable[[ProjectConfiguration, str | None], None]]
    ] = PrivateAttr(default_factory=dict)
    _claude_mcp_oauth_token_refresh_lock: Lock = PrivateAttr(default_factory=Lock)
    _claude_mcp_oauth_last_token_refresh_time_seconds: dict[str, float] = PrivateAttr(default_factory=dict)
    _claude_mcp_oauth_token_refresh_stop_event: Event = PrivateAttr(default_factory=Event)

    def _make_canonical(self, workspace_path: Path) -> Path:
        return workspace_path.expanduser().resolve()

    @property
    def _claude_dir(self) -> Path:
        return self.claude_root_path_on_users_local_machine / CLAUDE_DIRECTORY

    def start(self) -> None:
        self.secret_file_path.parent.mkdir(parents=True, exist_ok=True)
        try:
            credentials = Credentials.model_validate_json(self.credentials_file_path.read_text())
            if credentials.anthropic:
                self.set_anthropic_credentials(anthropic_credentials=credentials.anthropic)
            if credentials.openai:
                self.set_openai_credentials(openai_credentials=credentials.openai)
        except (FileNotFoundError, ValidationError):
            user_config = get_user_config_instance()
            if user_config:
                if user_config.anthropic_api_key:
                    self.set_anthropic_credentials(
                        AnthropicApiKey(
                            anthropic_api_key=Secret(user_config.anthropic_api_key), generated_from_oauth=False
                        )
                    )
                # TODO(Andy): Investigate if this is a no-op
                if user_config.openai_api_key:
                    self.set_openai_credentials(
                        OpenAIApiKey(openai_api_key=Secret(user_config.openai_api_key), generated_from_oauth=False)
                    )

        self._observer = Observer()
        try:
            self._claude_dir.mkdir(parents=True, exist_ok=True)
            self.load_claude_global_settings()
            self.load_agent_configurations()
            self.load_custom_command_configurations()
            self._observer.schedule(_ClaudeGlobalSettingsChangeHandler(self), str(self._claude_dir), recursive=True)
            self._observer.start()
        except OSError as e:
            logger.error(f"Failed to create or watch ~/.claude directory: {e}")
        except Exception as e:
            logger.error(f"Unexpected error setting up ~/.claude watcher: {e}")

    def stop(self) -> None:
        if self.token_refresh_thread:
            self._stop_token_refresh_thread()

        if self._observer:
            self._observer.stop()
            self._observer.join()
            self._observer = None

        self._claude_mcp_oauth_token_refresh_stop_event.set()

    def get_credentials(self) -> Credentials:
        return self.credentials

    def set_anthropic_credentials(self, anthropic_credentials: AnthropicCredentials) -> None:
        old_credentials_is_claude_oauth = isinstance(self.credentials.anthropic, ClaudeOauthCredentials)
        new_credentials_is_claude_oauth = isinstance(anthropic_credentials, ClaudeOauthCredentials)
        if old_credentials_is_claude_oauth and not new_credentials_is_claude_oauth:
            self._stop_token_refresh_thread()
        self.credentials = Credentials(anthropic=anthropic_credentials, openai=self.credentials.openai)
        if isinstance(anthropic_credentials, ClaudeOauthCredentials):
            self._on_new_user_config(GlobalConfiguration(credentials=self.credentials))
        populate_credentials_file(path=self.credentials_file_path, credentials=self.credentials)
        if not old_credentials_is_claude_oauth and new_credentials_is_claude_oauth:
            self._start_token_refresh_thread()

    def set_openai_credentials(self, openai_credentials: OpenAIApiKey) -> None:
        self.credentials = Credentials(openai=openai_credentials, anthropic=self.credentials.anthropic)
        populate_credentials_file(path=self.credentials_file_path, credentials=self.credentials)

    def remove_anthropic_credentials(self) -> None:
        self.credentials = Credentials(anthropic=None, openai=self.credentials.openai)
        self._write_credentials()

    def remove_openai_credentials(self) -> None:
        self.credentials = Credentials(openai=None, anthropic=self.credentials.anthropic)
        self._write_credentials()

    def _write_credentials(self) -> None:
        if self.credentials.is_set:
            populate_credentials_file(path=self.credentials_file_path, credentials=self.credentials)
        else:
            try:
                self.credentials_file_path.unlink()
            except FileNotFoundError:
                pass

    def _start_token_refresh_thread(self) -> None:
        self.token_refresh_thread = self.concurrency_group.start_new_thread(target=self._token_refresh_thread_target)

    def _stop_token_refresh_thread(self) -> None:
        self.token_refresh_stop_event.set()
        self.token_refresh_thread.join()  # pyre-fixme[16]: token_refresh_thread can be None
        self.token_refresh_thread = None
        self.token_refresh_stop_event = Event()

    def _token_refresh_thread_target(self) -> None:
        first_iteration = True
        while not self.concurrency_group.is_shutting_down():
            if first_iteration:
                first_iteration = False
            else:
                # Wait for a short time between all iterations,
                # but not before the first iteration -
                # the OAuth token might already have expired when Sculptor starts.
                #
                # The timeout may seem unnecessarily short short,
                # as the token is usually valid for at least a couple of hours.
                # However, the user's computer could go to sleep and we can overshoot the expiry.
                # Minimize that possiblity by checking more frequently.
                should_stop = self.token_refresh_stop_event.wait(timeout=30)
                if should_stop:
                    break
            logger.debug("Claude OAuth token refresh thread has woken up")
            anthropic_credentials = self.credentials.anthropic

            # NOTE(bowei): very important not to throw exception here, cuz this flow can race with the re-login oauth modal flow.
            # Instead the token refresh thread should wait around until the relogin has finished. If we error here it will kill the api request too!
            if not isinstance(anthropic_credentials, ClaudeOauthCredentials):
                continue
            if time.time() < anthropic_credentials.expires_at_unix_ms / 1000 - REFRESH_TOKEN_EXPIRY_BUFFER_SECONDS:
                continue
            logger.info("Refreshing Claude OAuth tokens")
            refresh_token = anthropic_credentials.refresh_token.unwrap()
            with httpx.Client() as client:
                raw_response: httpx.Response | None = None
                try:
                    raw_response = client.post(
                        "https://console.anthropic.com/v1/oauth/token",
                        data={
                            "grant_type": "refresh_token",
                            "refresh_token": refresh_token,
                            "client_id": CLAUDE_CODE_CLIENT_ID,
                        },
                        headers={"Accept": "application/json"},
                    )
                    token_response = TokenResponse.model_validate_json(raw_response.content)
                except Exception as e:
                    log_exception(e, "Error refreshing Claude OAuth credentials")
                    # If we have failed, the response wouldn't contain any secret credentials,
                    # so it's safe to log.
                    if raw_response is not None:
                        logger.info("Raw response: {}", raw_response.content)
                    logger.info("Ignoring the error; we'll try again later")
                    continue
            self.credentials = Credentials(
                anthropic=ClaudeOauthCredentials(
                    access_token=Secret(token_response.access_token),
                    refresh_token=Secret(token_response.refresh_token),
                    expires_at_unix_ms=int((time.time() + token_response.expires_in) * 1000),
                    scopes=token_response.scope.split(" "),
                    subscription_type=anthropic_credentials.subscription_type,
                ),
                openai=self.credentials.openai,
            )
            populate_credentials_file(path=self.credentials_file_path, credentials=self.credentials)
            self._on_new_user_config(GlobalConfiguration(credentials=self.credentials))

    def _on_new_user_config(self, user_config: GlobalConfiguration) -> None:
        self.notify_global_anthropic_configuration_watchers()

    def get_user_secrets(self, secret_names: Sequence[str] | None = None) -> dict[str, Secret]:
        file_secrets = {}
        if self.secret_file_path.exists():
            file_secrets = dotenv_values(self.secret_file_path)

        secrets = file_secrets
        if os.getenv(GITLAB_TOKEN_NAME) is not None:
            # NOTE(bowei): this is NOT supposed to be a user-owned key. TODO(PROD-3226): rename the name to be clearer
            # Sculptor mirrors user repos to GitLab; this token must be forwarded so the backend keeps push access.
            secrets[GITLAB_TOKEN_NAME] = os.environ[GITLAB_TOKEN_NAME]

        if secret_names is not None:
            secrets = {name: secrets[name] for name in secret_names if name in secrets}

        secrets = {key: Secret(value) for key, value in secrets.items()}

        return secrets

    def set_user_secrets(self, secrets: Mapping[str, str | Secret]) -> None:
        logger.debug("Saving {} secrets to {}", len(secrets), self.secret_file_path)

        self.secret_file_path.parent.mkdir(parents=True, exist_ok=True)

        for key, value in secrets.items():
            set_key(
                dotenv_path=str(self.secret_file_path),
                key_to_set=key,
                value_to_set=value.unwrap() if isinstance(value, Secret) else value,
                quote_mode="auto",
            )

    def get_user_config(self) -> UserConfig:
        return get_user_config_instance()

    def set_user_config(self, config: UserConfig) -> None:
        original_config = get_user_config_instance()
        config_path = get_config_path()
        save_config(config, config_path)
        set_user_config_instance(config)
        if config.is_claude_configuration_synchronized and not original_config.is_claude_configuration_synchronized:
            # When the user enables Claude settings synchronization, we need to make sure the settings get propagated to existing tasks.
            self.notify_global_anthropic_configuration_watchers()
            for workspace_key in self._project_configuration_watchers.keys():
                self.notify_project_configuration_watchers(Path(workspace_key))

    def load_claude_global_settings(self) -> bool:
        settings_file = self._claude_dir / CLAUDE_GLOBAL_SETTINGS_FILENAME
        original_settings = self._claude_global_settings
        with self._settings_lock:
            if not settings_file.exists():
                self._claude_global_settings = None
            else:
                try:
                    self._claude_global_settings = merge_tool_readiness_hook(
                        _preprocess_claude_settings_json(json.loads(settings_file.read_text())), timeout_seconds=120
                    )
                except json.JSONDecodeError:
                    # TODO(andrew.laack): We likely want to surface this error to users.
                    self._claude_global_settings = None
        return original_settings != self._claude_global_settings

    def load_claude_local_settings(self, workspace_path: Path) -> bool:
        workspace_path = self._make_canonical(workspace_path)
        settings_file = workspace_path / CLAUDE_DIRECTORY / CLAUDE_LOCAL_SETTINGS_FILENAME
        original_settings = self._claude_local_settings.get(str(workspace_path))
        with self._settings_lock:
            if not settings_file.exists():
                self._claude_local_settings[str(workspace_path)] = None
            else:
                try:
                    self._claude_local_settings[str(workspace_path)] = _preprocess_claude_settings_json(
                        json.loads(settings_file.read_text())
                    )
                except json.JSONDecodeError:
                    # TODO(andrew.laack): Notify users of this.
                    self._claude_local_settings[str(workspace_path)] = None
        return original_settings != self._claude_local_settings[str(workspace_path)]

    def load_agent_configurations(self) -> bool:
        original_subagents = self._claude_global_subagents
        agents_dir = self._claude_dir / SUBAGENTS_DIRECTORY
        if not agents_dir.exists():
            self._claude_global_subagents = None
        else:
            subagents = {}
            for agent_file in agents_dir.rglob("*.md"):
                agent_filename, agent_definition = agent_file.name, agent_file.read_text()
                with self._settings_lock:
                    subagents[agent_filename] = agent_definition
            self._claude_global_subagents = subagents
        return original_subagents != self._claude_global_subagents

    def load_custom_command_configurations(self) -> bool:
        commands_dir = self._claude_dir / COMMANDS_DIRECTORY
        original_commands = self._claude_global_custom_commands
        if not commands_dir.exists():
            self._claude_global_custom_commands = None
        else:
            custom_commands = {}
            for command_file in commands_dir.rglob("*.md"):
                command_path, command_definition = command_file.relative_to(commands_dir), command_file.read_text()
                with self._settings_lock:
                    custom_commands[str(command_path)] = command_definition
            self._claude_global_custom_commands = custom_commands
        return original_commands != self._claude_global_custom_commands

    def get_claude_root_path_on_users_local_machine(self) -> Path:
        return self.claude_root_path_on_users_local_machine

    def get_global_configuration(self) -> GlobalConfiguration:
        with self._settings_lock:
            claude_config = ClaudeGlobalConfiguration(
                settings=self._claude_global_settings.copy() if self._claude_global_settings is not None else None,
                subagents=self._claude_global_subagents.copy() if self._claude_global_subagents is not None else None,
                custom_commands=self._claude_global_custom_commands.copy()
                if self._claude_global_custom_commands is not None
                else None,
            )

            return GlobalConfiguration(
                credentials=self.credentials,
                claude_config=claude_config,
            )

    def register_global_configuration_watcher(
        self, callback: Callable[[GlobalConfiguration, str | None], None]
    ) -> None:
        self._anthropic_global_configuration_watchers.append(callback)

    def notify_global_anthropic_configuration_watchers(self, update_description: str | None = None) -> None:
        anthropic_configuration = self.get_global_configuration()

        for callback in self._anthropic_global_configuration_watchers:
            try:
                callback(anthropic_configuration, update_description)
            except Exception as e:
                logger.error(f"Error in user config watcher callback: {e}")

    def get_project_configuration(self, workspace_path: Path) -> ProjectConfiguration | None:
        workspace_key = str(self._make_canonical(workspace_path))
        with self._settings_lock:
            if workspace_key not in self._project_configuration_watchers:
                return
            project_id, _ = self._project_configuration_watchers[workspace_key]
            settings = self._claude_local_settings.get(workspace_key)
            return ProjectConfiguration(claude_workspace_settings=settings)

    def notify_project_configuration_watchers(
        self, workspace_path: Path, update_description: str | None = None
    ) -> None:
        workspace_path = self._make_canonical(workspace_path)
        project_configuration = self.get_project_configuration(workspace_path)
        if project_configuration is None:
            return
        project_id, callback = self._project_configuration_watchers[str(workspace_path)]
        try:
            callback(project_configuration, update_description)
        except Exception as e:
            logger.error(f"Error in project workspace watcher callback for project {project_id}: {e}")

    def register_project_configuration_watcher(
        self,
        project_id: "ProjectID",
        workspace_path: Path,
        callback: Callable[[ProjectConfiguration, str | None], None],
    ) -> None:
        workspace_key = str(self._make_canonical(workspace_path))
        if workspace_key not in self._project_configuration_watchers:
            claude_dir = workspace_path / CLAUDE_DIRECTORY
            try:
                claude_dir.mkdir(parents=True, exist_ok=True)
                self.load_claude_local_settings(workspace_path)
                self._observer.schedule(_ClaudeLocalSettingsChangeHandler(self), str(claude_dir), recursive=False)
            except OSError as e:
                logger.error(f"Failed to create or watch .claude directory at {claude_dir}: {e}")
            except Exception as e:
                logger.error(f"Unexpected error setting up watcher for {claude_dir}: {e}")

        with self._settings_lock:
            self._project_configuration_watchers[workspace_key] = (project_id, callback)

        self.notify_project_configuration_watchers(workspace_path)

    def maybe_refresh_claude_mcp_oauth_tokens(self, workspace_path: Path) -> None:
        workspace_path = self._make_canonical(workspace_path)
        # TODO: surface various failures to the user somehow.
        if self._claude_mcp_oauth_token_refresh_stop_event.is_set():
            return
        with self._claude_mcp_oauth_token_refresh_lock:
            credentials_data = get_local_credentials_data(self)
            if credentials_data is None:
                return
            last_refreshed_at = self._claude_mcp_oauth_last_token_refresh_time_seconds.get(str(workspace_path))
            now = datetime.now()
            # Is this refresh coming soon after the last one?
            is_frequent = last_refreshed_at is not None and (
                time.monotonic() - last_refreshed_at < CLAUDE_MCP_OAUTH_FREQUENCY_THRESHOLD_SECONDS
            )
            for server_name, server_details in credentials_data.get("mcpOAuth", {}).items():
                expires_at_miliseconds = server_details.get("expiresAt")
                if not expires_at_miliseconds:
                    continue
                expires_at = datetime.fromtimestamp(expires_at_miliseconds / 1000)
                if expires_at > now + timedelta(seconds=CLAUDE_MCP_OAUTH_BUFFER_LONG_SECONDS):
                    continue
                if expires_at > now + timedelta(seconds=CLAUDE_MCP_OAUTH_BUFFER_SHORT_SECONDS) and is_frequent:
                    continue
                break
            else:
                # No refresh needed.
                return
            try:
                # This call updates .credentials.json in place - the tokens are renewed if necessary.
                process = self.concurrency_group.run_process_to_completion(
                    ["claude", "mcp", "list"],
                    shutdown_event=self._claude_mcp_oauth_token_refresh_stop_event,
                    # Needs to run in the workspace so that claude can identify the right set of MCP servers.
                    cwd=workspace_path,
                )
                # TODO: check for the "Needs authentication" line in the output and tell the user.
            except ProcessSetupError:
                logger.info(
                    "Could not refresh Claude MCP OAuth tokens because claude code does not seem to be installed."
                )
                return
            self._claude_mcp_oauth_last_token_refresh_time_seconds[str(workspace_path)] = time.monotonic()


# TODO: Let users know if their settings.json contained unsupported fields that were ignored.
def _preprocess_claude_settings_json(settings_json: dict[str, Any]) -> dict[str, Any]:
    """
    Preprocess the host Claude settings.json.

    Removes fields that should not be transferred to Sculptor sandboxes.

    """
    preprocessed = {**settings_json}
    # apiKeyHelper points to a script that typically won't exist in a container and thus can break oauth.
    preprocessed.pop("apiKeyHelper", None)
    return preprocessed
