import json
import pathlib
import shlex
import tempfile
from pathlib import Path
from typing import Any
from typing import Final
from typing import Mapping
from typing import Sequence
from typing import assert_never

from loguru import logger

from imbue_core.sculptor.state.mcp_constants import IMBUE_CLI_INTERNAL_MCP_SERVER_NAME
from imbue_core.sculptor.state.mcp_constants import IMBUE_CLI_USER_MCP_SERVER_NAME
from sculptor.interfaces.environments.base import Environment
from sculptor.services.config_service.data_types import AWSBedrockApiKey
from sculptor.services.config_service.data_types import AnthropicApiKey
from sculptor.services.config_service.data_types import AnthropicCredentials
from sculptor.services.config_service.data_types import ClaudeOauthCredentials
from sculptor.services.environment_service.tool_readiness import READY_FILE

# Number of characters from the end of API key to store for approval tracking
API_KEY_SUFFIX_LENGTH = 20


def populate_claude_settings(environment: Environment, anthropic_credentials: AnthropicCredentials) -> None:
    """Claude Code requires certain settings to run correctly.

    We default to using the user's settings (with some specific changes).
    However, if the user does NOT have claude code installed, we can provide
    them with our own settings.
    """
    logger.info("Populating claude settings")
    claude_config_path = Path.home() / ".claude.json"

    if claude_config_path.exists():
        logger.info("Found existing claude config path at {}", claude_config_path)
        claude_config = json.load(open(claude_config_path, "r"))

        # Make required modifications
        claude_config["projects"][str(environment.to_host_path(environment.get_workspace_path()))] = (
            _claude_project_config(environment.to_host_path(environment.get_workspace_path()))
        )
    else:
        logger.info("Generating new claude config")
        claude_config = _claude_config_template(environment)
    # A previous version of this code set the primaryApiKey,
    # but we have since moved to injecting environment variables.
    # Remove it in case the container has an old config
    # and there's a conflict between the two approaches.
    claude_config.pop("primaryApiKey", None)
    credentials_file_path = environment.get_container_user_home_directory() / ".claude" / ".credentials.json"
    credentials_tmp_file_path = environment.get_container_user_home_directory() / ".claude" / ".credentials.json.tmp"
    match anthropic_credentials:
        case AnthropicApiKey(anthropic_api_key=anthropic_api_key):
            # this is required for claude to work with the anthropic api key without prompting the user (primarily required for compaction and terminal)
            claude_config["customApiKeyResponses"] = {
                "approved": [anthropic_api_key.unwrap()[-API_KEY_SUFFIX_LENGTH:]],
                "rejected": [],
            }
            logger.trace("Writing anthropic api key to {}", credentials_file_path)
            # TODO: This works but it's safer to remove the file completely.
            environment.write_atomically(str(credentials_file_path), ".tmp", "")
            logger.trace("Wrote anthropic api key to {}", credentials_file_path)
        case ClaudeOauthCredentials():
            claude_config["customApiKeyResponses"] = {
                "approved": [],
                "rejected": [],
            }
            logger.trace("Writing claude oauth credentials to {}", credentials_file_path)
            environment.write_atomically(
                str(credentials_file_path), ".tmp", anthropic_credentials.convert_to_claude_code_credentials_json()
            )
            logger.trace("Wrote claude oauth credentials to {}", credentials_file_path)
        case AWSBedrockApiKey(bedrock_api_key=bedrock_api_key):
            claude_config["customApiKeyResponses"] = {
                "approved": [bedrock_api_key.unwrap()[-API_KEY_SUFFIX_LENGTH:]],
                "rejected": [],
            }
            logger.trace("Writing bedrock api key to {}", credentials_file_path)
            # TODO: This works but it's safer to remove the file completely.
            environment.write_file(str(credentials_file_path), "")
            logger.trace("Wrote bedrock api key to {}", credentials_file_path)
        case _ as unreachable:
            # TODO: pyre doesn't understand the matching here
            assert_never(unreachable)  # pyre-fixme[6]
    claude_config["hasCompletedOnboarding"] = True

    with tempfile.NamedTemporaryFile() as tmp_file:
        claude_config_path = environment.get_container_user_home_directory() / ".claude.json"
        logger.trace("Writing claude config to environment at {}", claude_config_path)
        environment.write_atomically(str(claude_config_path), ".tmp", json.dumps(claude_config))

    logger.info("Populated claude settings")


def _claude_mcp_servers_config(workspace_path: pathlib.Path) -> dict[str, Any]:
    return {
        name: {
            "command": "imbue-cli.sh",
            "args": [
                "--log-to-file=/tmp/imbue-cli.log",
                "mcp",
                *("--project-path", str(workspace_path)),
                *config_args,
                *("--transport", "stdio"),
            ],
            "env": {},
        }
        for name, config_args in (
            (IMBUE_CLI_INTERNAL_MCP_SERVER_NAME, ["--use-internal-config"]),
            (IMBUE_CLI_USER_MCP_SERVER_NAME, ["--config", str(workspace_path / "tools.toml")]),
        )
    }


def _claude_project_config(workspace_path: pathlib.Path) -> dict[str, Any]:
    # TODO: do we need all of these settings? last session id seems to be randomly copy pasted from someone's .claude.json
    return {
        "allowedTools": [],
        "history": [],
        "dontCrawlDirectory": False,
        "mcpContextUris": [],
        "mcpServers": _claude_mcp_servers_config(workspace_path),
        "enabledMcpjsonServers": [],
        "disabledMcpjsonServers": [],
        "hasTrustDialogAccepted": True,
        "ignorePatterns": [],
        "projectOnboardingSeenCount": 1,
        "hasClaudeMdExternalIncludesApproved": False,
        "hasClaudeMdExternalIncludesWarningShown": False,
        "lastCost": 0,
        "lastAPIDuration": 0,
        "lastDuration": 3172,
        "lastLinesAdded": 0,
        "lastLinesRemoved": 0,
        "lastTotalInputTokens": 0,
        "lastTotalOutputTokens": 0,
        "lastTotalCacheCreationInputTokens": 0,
        "lastTotalCacheReadInputTokens": 0,
        "lastSessionId": "ef949ec0-4a45-4665-81a7-f9e1ec21a41c",
        "bypassPermissionsModeAccepted": True,
    }


def _claude_config_template(environment: Environment) -> dict[str, Any]:
    return {
        "numStartups": 3,
        "theme": "light",
        "customApiKeyResponses": {
            "approved": [],
            "rejected": [],
        },
        "firstStartTime": "2025-06-10T21:50:05.520Z",
        "projects": {
            str(environment.to_host_path(environment.get_workspace_path())): _claude_project_config(
                environment.to_host_path(environment.get_workspace_path())
            )
        },
        "isQualifiedForDataSharing": False,
        "hasCompletedOnboarding": True,
        "lastOnboardingVersion": "1.0.17",
        "recommendedSubscription": "",
        "subscriptionNoticeCount": 0,
        "hasAvailableSubscription": False,
    }


def _read_existing_settings(environment: Environment, settings_path: pathlib.Path) -> dict[str, Any]:
    """Read existing settings.json file if it exists.

    Args:
        environment: The environment to read from
        settings_path: Path to the settings.json file

    Returns:
        Existing settings, or empty dict if file doesn't exist

    Raises:
        Exception: If an unexpected error occurs (not FileNotFoundError)
        json.JSONDecodeError: If the existing settings file has invalid JSON
    """
    try:
        content = environment.read_file(str(settings_path))
        logger.debug("Read existing settings file from {}", settings_path)
    except FileNotFoundError:
        logger.debug("No existing settings file at {}", settings_path)
        return {}

    settings = json.loads(content)
    logger.debug("Parsed settings with {} top-level keys", len(settings))
    return settings


_HOOK_SCRIPT_PATH: Final[str] = "/imbue/bin/check_tool_readiness.sh"


def _matches_tool_readiness_hook(hook_config: object) -> bool:
    """Check if a given hook configuration matches our tool readiness hook."""

    # Imperative logic to check if the structure looks something like this:
    # {
    #   "matcher": "*",
    #   "hooks": [
    #     {
    #       "type": "command",
    #       "command": "/imbue/bin/check_tool_readiness.sh <ready_file_path>",
    #       ...,
    #     },
    #   ],
    # }
    #
    # Since this can contain arbitrary user-defined data, we have to be defensive.

    if not isinstance(hook_config, Mapping):
        return False
    matcher = hook_config.get("matcher")
    hooks = hook_config.get("hooks")
    if matcher != "*":
        return False
    if not isinstance(hooks, Sequence) or len(hooks) == 0:
        return False
    first_hook = hooks[0]
    if not isinstance(first_hook, Mapping):
        return False
    command = first_hook.get("command")
    if not isinstance(command, str):
        return False
    return _HOOK_SCRIPT_PATH in command


def _merge_tool_readiness_hook(existing_settings: Mapping[str, Any], timeout_seconds: int) -> dict[str, Any]:
    """Merge tool readiness hook configuration with existing settings.

    Preserves all existing settings and hooks, appending the tool readiness PreToolUse hook.
    Multiple hooks with the same matcher can coexist and will all run in parallel.

    Args:
        existing_settings: Existing settings
        timeout_seconds: Timeout for the tool readiness hook

    Returns:
        New settings dictionary with hook appended

    Raises:
        TypeError: If existing settings structure has unexpected types
    """
    our_hook_config = {
        "matcher": "*",
        "hooks": [
            {
                "type": "command",
                "command": f"{_HOOK_SCRIPT_PATH} {shlex.quote(READY_FILE.as_posix())}",
                "env": {
                    "SCULPTOR_TOOL_READINESS_TIMEOUT": str(timeout_seconds),
                },
            }
        ],
    }

    existing_hooks = existing_settings.get("hooks", {})
    if not isinstance(existing_hooks, dict):
        raise TypeError(f"Expected hooks to be a dict, got {type(existing_hooks).__name__}")

    existing_pre_tool_use = existing_hooks.get("PreToolUse", [])
    if not isinstance(existing_pre_tool_use, (list, tuple)):
        raise TypeError(f"Expected PreToolUse to be a list, got {type(existing_pre_tool_use).__name__}")

    logger.debug("Appending tool readiness hook to PreToolUse hooks")

    return {
        **existing_settings,
        "hooks": {
            **existing_hooks,
            "PreToolUse": [
                *(
                    hook_config
                    for hook_config in existing_pre_tool_use
                    if not _matches_tool_readiness_hook(hook_config)
                ),
                our_hook_config,
            ],
        },
    }


def configure_tool_readiness_hook(environment: Environment, timeout_seconds: int = 120) -> None:
    """Configure the PreToolUse hook to block tool execution until environment is ready.

    Merges the hook configuration with any existing settings.json, preserving all
    other settings and hooks.

    Args:
        environment: The environment to configure
        timeout_seconds: Maximum seconds to wait before timing out
    """
    logger.info("Configuring tool readiness hook (timeout: {}s)", timeout_seconds)

    settings_dir = environment.get_container_user_home_directory() / ".claude"
    settings_path = settings_dir / "settings.json"

    existing_settings = _read_existing_settings(environment, settings_path)
    if not isinstance(existing_settings, dict):
        raise TypeError(f"Expected existing settings to be a dict, got {type(existing_settings).__name__}")
    merged_settings = _merge_tool_readiness_hook(existing_settings, timeout_seconds)

    with tempfile.NamedTemporaryFile(mode="w", suffix=".json") as tmp_file:
        json.dump(merged_settings, tmp_file, indent=2)
        tmp_file.flush()

        environment.run_process_to_completion(["mkdir", "-p", str(settings_dir)], is_checked=True, secrets={})
        environment.copy_from_local(Path(tmp_file.name), str(settings_path))
        logger.info("Tool readiness hook configured successfully at {}", settings_path)
