from loguru import logger

from imbue_core.sculptor.telemetry import PosthogEventModel
from imbue_core.sculptor.telemetry import PosthogEventPayload
from imbue_core.sculptor.telemetry import emit_posthog_event
from imbue_core.sculptor.telemetry import with_consent
from imbue_core.sculptor.telemetry_constants import ConsentLevel
from imbue_core.sculptor.telemetry_constants import ProductComponent
from imbue_core.sculptor.telemetry_constants import SculptorPosthogEvent
from sculptor.agents.default.constants import AGENT_RESPONSE_TYPE_TO_POSTHOG_EVENT_MAP
from sculptor.agents.default.constants import USER_MESSAGE_TYPE_TO_POSTHOG_EVENT_MAP
from sculptor.database.models import TaskID
from sculptor.interfaces.agents.agent import ParsedAgentResponseType
from sculptor.interfaces.agents.agent import UserMessageUnion


def emit_posthog_event_for_agent_message(task_id: TaskID, message: ParsedAgentResponseType) -> None:
    if message.object_type not in AGENT_RESPONSE_TYPE_TO_POSTHOG_EVENT_MAP:
        logger.error(
            "Unknown object type '{}' in _emit_posthog_event_for_agent_message. If you've added a new message type to ParsedAgentResponseType, please update AGENT_RESPONSE_TYPE_TO_POSTHOG_EVENT_MAP.",
            message.object_type,
        )
        return

    posthog_event = AGENT_RESPONSE_TYPE_TO_POSTHOG_EVENT_MAP[message.object_type]

    emit_posthog_event(
        PosthogEventModel(
            name=posthog_event, component=ProductComponent.CLAUDE_CODE, task_id=str(task_id), payload=message
        )
    )


def emit_posthog_event_for_user_message(task_id: TaskID, message: UserMessageUnion) -> None:
    if message.object_type not in USER_MESSAGE_TYPE_TO_POSTHOG_EVENT_MAP:
        logger.error(
            "Unknown object type '{}' in _emit_posthog_event_for_user_message. If you've added a new message type to UserMessageUnion, please update USER_MESSAGE_TYPE_TO_POSTHOG_EVENT_MAP.",
            message.object_type,
        )
        return

    posthog_event = USER_MESSAGE_TYPE_TO_POSTHOG_EVENT_MAP[message.object_type]

    emit_posthog_event(
        PosthogEventModel(
            name=posthog_event, component=ProductComponent.CLAUDE_CODE, task_id=str(task_id), payload=message
        )
    )


class ClaudeCommandLog(PosthogEventPayload):
    command: list[str] = with_consent(ConsentLevel.LLM_LOGS, default=[])
    system_prompt: str = with_consent(ConsentLevel.LLM_LOGS, default="")
    user_instructions: str = with_consent(ConsentLevel.LLM_LOGS, default="")


def emit_posthog_claude_command_event(
    task_id: TaskID, command: list[str], system_prompt: str, user_instructions: str
) -> None:
    emit_posthog_event(
        PosthogEventModel(
            name=SculptorPosthogEvent.CLAUDE_COMMAND,
            component=ProductComponent.CLAUDE_CODE,
            task_id=str(task_id),
            payload=ClaudeCommandLog(
                command=command, system_prompt=system_prompt, user_instructions=user_instructions
            ),
        )
    )
