"""Downloads the Imbue control plane from ghcr.io and makes it available as Docker volumes.

TODO: MAJOR: Figure out a plan for how we're going to clean up these Docker volumes when we move to the next version.

This is very much an optimization to prevent paying a "layer tax" for the control plane.

The idea here is that for content that doesn't change often, we can just fetch it from the image once,
copy it into a volume, and attach it to containers as a read-only volume, rather than a layer.


The alternative would be something like this inside Dockerfile.imbue_addons, which would
copy the control plane directly into the image:
```
ARG SCULPTORBASE_NIX=ghcr.io/imbue-ai/sculptorbase_nix:...
FROM ${SCULPTORBASE_NIX} AS sculptorbase_nix
FROM ubuntu:24.04
COPY --from=sculptorbase_nix /imbue /imbue
```

But the problems with layers are:
1. Changing them invalidates all subsequent layers, and you have to decide an ordering.
2. They're slow to build.

Doing it the "volume mounted" way enables:
* Shorter image build and export times, because the control plane isn't actually a layer in the user's image.
* Theoretically, we can "swap out" the control plane for a newer version without rebuilding the image, but just attaching a different volume.
"""

import json
import os
from enum import Enum
from enum import auto
from importlib.resources.abc import Traversable
from typing import Final

from loguru import logger

from imbue_core.background_setup import BackgroundSetup
from imbue_core.concurrency_group import ConcurrencyGroup
from imbue_core.itertools import only
from imbue_core.processes.local_process import run_blocking
from imbue_core.pydantic_serialization import SerializableModel
from imbue_core.subprocess_utils import ProcessError
from sculptor.primitives.constants import CONTROL_PLANE_LOCAL_TAG_PATH
from sculptor.primitives.constants import CONTROL_PLANE_MANIFEST_PATH
from sculptor.primitives.constants import CONTROL_PLANE_TAG_PATH
from sculptor.services.environment_service.providers.docker.image_fetch import ImagePurpose
from sculptor.services.environment_service.providers.docker.image_fetch import fetch_image_from_cdn
from sculptor.services.environment_service.providers.docker.image_utils import get_platform_architecture
from sculptor.utils.timeout import log_runtime_decorator


# Docker volume mount arguments for control plane volumes
def get_control_plane_volume_docker_args() -> tuple[str, ...]:
    return (
        # Mount the volume as read-only to safely make the same volume available to multiple images.
        *("-v", f"{ControlPlaneImageNameProvider().get_control_plane_volume_name()}:/imbue:ro"),
    )


class ControlPlaneFetchError(Exception):
    pass


class ControlPlaneRunMode(Enum):
    # Run a control plane version that has already been built locally, e.g. for running tests in CI.
    LOCALLY_BUILT = auto()

    # Run a control plane version that has been built and published, e.g. as in production.
    TAGGED_RELEASE = auto()


class ControlPlaneImageNameProvider(SerializableModel):
    """Provides the control plane image name based on the current run mode.

    Useful to parametrize some of the constants for testing.
    """

    control_plane_tag_path: Traversable = CONTROL_PLANE_TAG_PATH
    control_plane_local_tag_path: Traversable = CONTROL_PLANE_LOCAL_TAG_PATH
    control_plane_manifest_path: Traversable = CONTROL_PLANE_MANIFEST_PATH
    predetermined_run_mode: ControlPlaneRunMode | None = None

    def determine_current_run_mode(self) -> "ControlPlaneRunMode":
        """
        As of this writing, we still pin the control plane version to the tag specified in CONTROL_PLANE_TAG_PATH, and this file
        is tracked by git. By default, we want to use the tag specified here.

        However, if the control plane has been built locally, we write its tag to CONTROL_PLANE_LOCAL_TAG_PATH, which is git-ignored
        so will not get accidentally committed.  We use this tag if the file exists.
        """
        if self.predetermined_run_mode is not None:
            return self.predetermined_run_mode
        if self.control_plane_local_tag_path.is_file():
            return ControlPlaneRunMode.LOCALLY_BUILT
        else:
            return ControlPlaneRunMode.TAGGED_RELEASE

    def _get_control_plane_git_commit_hash(self) -> str:
        run_mode = self.determine_current_run_mode()
        tag_path_to_use = self.control_plane_tag_path
        if run_mode == ControlPlaneRunMode.LOCALLY_BUILT:
            tag_path_to_use = self.control_plane_local_tag_path

        return tag_path_to_use.read_text().strip()

    def _get_control_plane_sha_from_manifest_file(self) -> str:
        manifest_data = json.loads(CONTROL_PLANE_MANIFEST_PATH.read_text().strip())["manifests"]
        our_platform = get_platform_architecture()
        control_plane_entry = only(x for x in manifest_data if x["platform"]["architecture"] == our_platform)
        control_plane_sha: Final[str] = control_plane_entry["digest"].split("sha256:")[-1]

        return control_plane_sha

    def _get_control_plane_sha_from_local_docker_image(self, image_name: str) -> str:
        result = run_blocking(
            command=[
                "docker",
                "image",
                "inspect",
                "--format",
                "{{.Id}}",
                image_name,
            ]
        )
        image_id = result.stdout.strip()
        if image_id.startswith("sha256:"):
            return image_id.split("sha256:")[-1]
        else:
            return "unknown_sha256"

    def _get_control_plane_image_sha256(self) -> str:
        run_mode = self.determine_current_run_mode()
        if run_mode == ControlPlaneRunMode.LOCALLY_BUILT:
            # We use this to name the volume; if the image gets rebuilt, we want that to result in a different sha256 and thus a different volume.
            local_image_name = self.determine_control_plane_image_name()
            return self._get_control_plane_sha_from_local_docker_image(local_image_name)
        else:
            # Use the pinned sha256
            return self._get_control_plane_sha_from_manifest_file()

    def determine_control_plane_image_name(self) -> str:
        """Return the docker image name to use for the control plane."""
        run_mode = self.determine_current_run_mode()
        if run_mode == ControlPlaneRunMode.LOCALLY_BUILT:
            commit_hash = self._get_control_plane_git_commit_hash()
            return f"sculptorbase_nix:local_build_{commit_hash}"
        else:
            commit_hash = self._get_control_plane_git_commit_hash()

            # Pinning to a SHA lets Docker avoid a network call to check with ghcr.io if the tag has been updated.
            # See: https://github.com/orgs/imbue-ai/packages/container/package/sculptorbase_nix.
            control_plane_sha = self._get_control_plane_image_sha256()
            return f"ghcr.io/imbue-ai/sculptorbase_nix:{commit_hash}@sha256:{control_plane_sha}"

    def get_control_plane_volume_name(self) -> str:
        # There is a dev mode where the user can just specify a pre-created control plane volume.
        # We will eventually deprecate this (since we are move to a world where the image is always rebuilt locally), but for now we support it.
        volume_name_set_in_env = os.environ.get("SCULPTOR_CONTROL_PLANE_VOLUME")
        if volume_name_set_in_env:
            return volume_name_set_in_env

        # We keep each version of the control plane in its own volume.
        # It's nice that the same volume can be shared between images; these must be read-only, though.
        # TODO: These volumes will need to be garbage collected somehow.
        commit_hash = self._get_control_plane_git_commit_hash()
        control_plane_sha = self._get_control_plane_image_sha256()
        return f"imbue_control_plane_{commit_hash}_{control_plane_sha}"


@log_runtime_decorator()
def _fetch_control_plane_volume(concurrency_group: ConcurrencyGroup) -> None:
    """Fetches /imbue from the control plane image into a single volume.

    There's a race condition here.
    To summarize:
    * Two processes can start populating the volume at the same time, and copy all the same files into it.
    * But once one of them writes the VOLUME_READY.TXT file, all the files should have been written at least once.
    * However, the second process can still be copying files into the volume, and would "overwrite" with the same contents.
    * I talked this through with ChatGPT and convinced myself this is OK: https://chatgpt.com/share/68b090b9-b354-8004-a487-8a6f003d6dee
    * I've looked at Docker's volume auto-initialization and it doesn't handle the race well: https://imbue-ai.slack.com/archives/C06MFB87T4P/p1757356166569579?thread_ts=1757349096.985299&cid=C06MFB87T4P
    """
    # Try to fetch the control plane image from CDN first.
    image_to_use = ControlPlaneImageNameProvider().determine_control_plane_image_name()
    if ControlPlaneImageNameProvider().determine_current_run_mode() == ControlPlaneRunMode.TAGGED_RELEASE:
        fetch_image_from_cdn(image_to_use, ImagePurpose.CONTROL_PLANE, concurrency_group)

    control_plane_volume_name = ControlPlaneImageNameProvider().get_control_plane_volume_name()
    logger.info("Making sure {} volume exists.", control_plane_volume_name)

    command = f"""
    set -e
    if [ -f /imbue_volume/VOLUME_READY.TXT ]; then
        echo "_fetch_control_plane_volume: {control_plane_volume_name} already exists and is ready."
    else
        echo "_fetch_control_plane_volume: Initializing {control_plane_volume_name} volume, copying from /imbue to /imbue_volume..."

        # Copy /imbue contents to /imbue_volume/
        # /imbue/. means everything in the directory, including the ".venv" directory, which wouldn't match a * glob.
        rsync -a /imbue/. /imbue_volume/

        touch /imbue_volume/VOLUME_READY.TXT
        echo "_fetch_control_plane_volume: {control_plane_volume_name} finished rsync'ing from image into volume."
    fi
    """

    try:
        finished_process = concurrency_group.run_process_to_completion(
            command=[
                *("docker", "run", "--rm"),
                *("-v", f"{control_plane_volume_name}:/imbue_volume"),
                image_to_use,
                *("sh", "-c", command),
            ],
            on_output=lambda line, is_stderr: logger.debug(line.strip()),
        )
        logger.info(
            "Finished process to fetch volume_name={}: stdout={}, stderr={}",
            control_plane_volume_name,
            finished_process.stdout,
            finished_process.stderr,
        )
    except ProcessError as e:
        raise ControlPlaneFetchError(
            f"Failed to fetch control plane volume {control_plane_volume_name} from image {image_to_use}"
        ) from e


CONTROL_PLANE_FETCH_BACKGROUND_SETUP: Final[BackgroundSetup] = BackgroundSetup(
    "SculptorControlPlaneVolumeFetchBackgroundSetup",
    _fetch_control_plane_volume,
)
