import mimetypes
import time
from pathlib import Path
from threading import Event

from modal import Sandbox

from imbue_core.processes.remote_process import RemoteRunningProcess
from sculptor.cli.dev_commands.run_tests.sandboxing import launch_idempotent_process_in_sandbox
from sculptor.cli.dev_commands.run_tests.sandboxing import run_idempotent_process_in_sandbox

SHARED_BUCKET = "int8-shared-internal"
TEST_RESULTS_DIR = Path("test-results")
TEST_REPORT_XML = Path("pytest_junit.xml")
TEST_REPORT_HTML = Path("pytest_junit.html")
TEST_REPORT_COVERAGE = Path("pytest_cov.xml")


def upload_file(local_path: Path, s3_key: str, bucket: str, client):
    # Automatically determine content type from file extension
    content_type, _ = mimetypes.guess_type(str(local_path))
    extra_args = {}
    if content_type:
        extra_args["ContentType"] = content_type

    client.upload_file(str(local_path), bucket, s3_key, ExtraArgs=extra_args)


def test_data_url_for_s3_key(s3_key: str) -> str:
    return f"https://go.snake-blues.ts.net/shared/{s3_key}"


def upload_file_continually(client, local_path: Path, s3_key: str, stop_event: Event):
    while True:
        # we first check, then upload, to make sure that when we're done we do a final upload
        is_done = stop_event.is_set()
        if is_done:
            # wait just a little bit to make sure everything makes it into the files
            time.sleep(1.0)
        if local_path.exists():
            try:
                upload_file(local_path, s3_key, SHARED_BUCKET, client)
            except Exception as e:
                print(f"Failed to upload {local_path} to S3: {e}")
        if is_done:
            print(f"Uploaded final output file to {s3_key}")
            break
        stop_event.wait(10)


def start_docker(sandbox: Sandbox, sandbox_died_event: Event, keyfile: str = "modal_ssh_key") -> RemoteRunningProcess:
    docker_daemon = launch_idempotent_process_in_sandbox(
        sandbox,
        "rm -f /var/run/docker.pid /run/docker/containerd/containerd.pid /var/run/docker/containerd/containerd.pid /var/run/docker.sock && bash /start-dockerd.sh || (ip link delete docker0 && sleep 10 && bash /start-dockerd.sh)",
        sandbox_died_event,
        keyfile=keyfile,
    )

    # wait until docker is running (even though it really should be by this point)
    while True:
        _docker_info_check_exit, _stdout, _stderr = run_idempotent_process_in_sandbox(
            sandbox, ["docker", "system", "info"], sandbox_died_event, is_logged=False, keyfile=keyfile
        )
        if _docker_info_check_exit == 0:
            break
        if docker_daemon.poll() is not None:
            raise RuntimeError(
                "Docker daemon failed to start:\n" + docker_daemon.read_stdout() + "\n" + docker_daemon.read_stderr()
            )
        time.sleep(1.0)

    return docker_daemon
