# sculptor database

This module is responsible for storing all permanent application data.
It currently supports sqlite and postgresql databases.

We store all of our data in a dual form -- as a log of immutable events (in `${table_name}`,)
and as a materialized view of the current state (in `${table_name}_latest`.)
For the implementation of that logic, see [automanaged.py](automanaged.py).

For the exact schema definitions, see the [models.py](models.py) file.
All table classes inherit from `DatabaseModel`.

All classes in this file, and (recursively) imported by this file, will end up in the database.
All classes that do *not* inherit from `DatabaseModel` will end up being serialized as JSON.

Each table has an ID class that is specific to that table, and which inherits from `ObjectID`.
Each ID class specifies an (ideally unique) prefix / tag that allows us to quickly distinguish their string representations.
See `sculptor/primitives/ids.py` for more details.

Note that there are [3 different ways of mapping inheritance in SQLAlchemy](https://docs.sqlalchemy.org/en/20/orm/inheritance.html):

- single table inheritance – several types of classes are represented by a single table;
- concrete table inheritance – each type of class is represented by independent tables;
- joined table inheritance – the class hierarchy is broken up among dependent tables. Each class represented by its own table that only includes those attributes local to that class.

Plus one other alternative:

- dispatched json column data -- where each class is represented by a single table, but the data that changes is stored in a JSON column.

All of them have trade-offs, and the approach we use is case-specific:
- For `Task` and `SavedAgentMessage`,
  we use the dispatched json approach since it doesn't really make sense to migrate the data --
  it will eventually be completely agent-dependent anyway,
  and we'll want to think about how to gracefully deal with outdated data.


## Migrations

We use Alembic for schema migrations. For the locally running sculptor instances, we run migrations automatically at server startup. See the [original proposal](../docs/proposals/database_migrations.md) for an overview of the intentions.

### Tracking SQL schema changes.

When you make a SQL-schema-affecting change to a database model without creating a migration, the `test_there_are_no_missing_sql_schema_migrations()` test should fail.
When that happens, run `uv run python sculptor/scripts/bump_migrations <migration_message>`. Alembic should automatically generate the necessary migration. Please review it and if necessary, adjust manually. Try to keep dialect compatibility in mind (we're trying to support sqlite and postgres).


### Tracking JSON schema changes.

When a database model contains a field which itself is a json-serialized pydantic model (or their union), changing the structure of the json field needs to be accompanied with a data migration (we call these "json schema migrations").

Failing to do so should cause a failure of the `test_there_are_no_missing_json_schema_migrations()` test. When this test fails, you should:

- Run `uv run python sculptor/scripts/bump_migrations <migration_message>`
- Inspect the changes in `sculptor/database/alembic/frozen_pydantic_schemas.json`.
- Decide whether the changes actually need a data migration or not.
    - (Some changes are actually backwards compatible. For example: adding a field with a default value. In these cases, we don't need to add migrations.)
- When a data migration is needed, implement the necessary logic in the autogenerated migration file. Otherwise delete the file.
    - For an example json schema migration, see `sculptor/database/alembic/examples/json_schema_migration.py`.
    - Try to keep dialect compatibility in mind (we're trying to support sqlite and postgres).
- Finally, commit the migration file (if needed) as well as the updated `frozen_pydantic_schemas.json` (in any case).

### Testing migrations

You can test new migrations by running `uv run alembic -x dburl=sqlite:///~/.sculptor/database.db upgrade head` from the `sculptor/database` directory.
(Ideally, back up your original database first.)
