import json
import os
from pathlib import Path
from typing import Any

from pydantic import ValidationError

from sculptor.services.config_service.data_types import Credentials
from sculptor.services.config_service.plugin_system import ConfigServicePlugin
from sculptor.services.config_service.plugin_system import ConfigurationContext
from sculptor.services.config_service.plugin_system import ConfigurationRule
from sculptor.services.config_service.plugin_system import SANDBOX_HOME_PLACEHOLDER


def _convert_credentials_json_to_auth_json(context: ConfigurationContext["CodexConfigurationPlugin"]) -> str | None:
    synchronized_from = context.synchronized_from_file
    assert synchronized_from is not None, "Expected synchronized_from_file to be set."
    try:
        credentials = Credentials.model_validate_json(synchronized_from.read_text())
        openai_api_key = credentials.openai
    except (FileNotFoundError, ValidationError):
        openai_api_key = None
    if openai_api_key is None:
        # Fall back to environment variable for backward compatibility.
        openai_api_key_str = os.environ.get("OPENAI_API_KEY")
    else:
        openai_api_key_str = openai_api_key.openai_api_key.unwrap()
    return json.dumps({"OPENAI_API_KEY": openai_api_key_str})


def _populate_config_toml(context: ConfigurationContext["CodexConfigurationPlugin"]) -> str | None:
    return (
        "\n".join(
            [
                f'[projects."{context.workspace_path_sandbox}"]',
                f'trust_level = "trusted"',
                "",
                "[shell_environment_policy]",
                "ignore_default_excludes = true",
            ]
        )
        + "\n"
    )


class CodexConfigurationPlugin(ConfigServicePlugin):
    credentials_path_local: Path

    def model_post_init(self, context: Any) -> None:
        self.configuration_rules = (
            ConfigurationRule(
                name=".codex/auth.json",
                synchronize_from=self.credentials_path_local,
                synchronize_to=SANDBOX_HOME_PLACEHOLDER / ".codex" / "auth.json",
                augment_function=_convert_credentials_json_to_auth_json,
                is_notifying_on_updates=False,
            ),
            ConfigurationRule(
                name=".codex/config.toml",
                synchronize_from=None,
                synchronize_to=SANDBOX_HOME_PLACEHOLDER / ".codex" / "config.toml",
                augment_function=_populate_config_toml,
                is_notifying_on_updates=False,
            ),
        )


def get_plugin(credentials_path_local: Path) -> CodexConfigurationPlugin:
    return CodexConfigurationPlugin(credentials_path_local=credentials_path_local)
